/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.enums.SuiteAuthType;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.google.common.collect.Maps;
import cn.kinyun.wework.sdk.entity.suite.AdminList;
import cn.kinyun.wework.sdk.entity.suite.AuthInfo;
import cn.kinyun.wework.sdk.entity.suite.CorpToken;
import cn.kinyun.wework.sdk.entity.suite.PermanentCode;
import cn.kinyun.wework.sdk.entity.suite.PreAuthCode;
import cn.kinyun.wework.sdk.entity.suite.SuiteToken;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;

/**
 * @title SuiteApi
 * @author yanmaoyuan
 * @date 2021年2月24日
 * @version 1.0
 */
@Component
public class SuiteApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;
    
    @Value("${qyapi.suite.install}")
    private String wwInstallUrl;

    @Value("${qyapi.suite.get_suite_token}")
    private String wwGetSuiteToken;

    @Value("${qyapi.suite.get_pre_auth_code}")
    private String wwGetPreAuthCode;
    
    @Value("${qyapi.suite.set_session_info}")
    private String wwSetSessionInfo;
    
    @Value("${qyapi.suite.get_permanent_code}")
    private String wwGetPermanentCode;

    @Value("${qyapi.suite.get_auth_info}")
    private String wwGetAuthInfo;

    @Value("${qyapi.suite.get_corp_token}")
    private String wwGetCorpToken;
    
    @Value("${qyapi.suite.get_admin_list}")
    private String wwGetAdminList;

    /**
     * 企业应用授权链接
     * 
     * @param suiteId
     * @param preAuthCode
     * @param redirectUri
     * @param state
     * @return
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90597
     */
    @GenIgnore
    public String getInstallUrl(@NonNull String suiteId, @NonNull String preAuthCode, @NonNull String redirectUri, @GenNullable String state) {
        return MessageFormat.format(wwInstallUrl, suiteId, preAuthCode, redirectUri, state);
    }

    /**
     * 获取第三方应用凭证
     * 
     * @param suiteId 以ww或wx开头应用id（对应于旧的以tj开头的套件id）
     * @param suiteSecret 应用secret
     * @param suiteTicket 企业微信后台推送的ticket
     * @return 第三方应用凭证（suite_access_token）
     * @throws WeworkException
     * @see <a href="https://open.work.weixin.qq.com/api/doc/90001/90143/90600">获取第三方应用凭证</a>
     * @see <a href="https://open.work.weixin.qq.com/api/doc/10982#%E6%8E%A8%E9%80%81suite_ticket">推送suite_ticket</a>
     */
    @GenIgnore
    public SuiteToken getSuiteToken(@NonNull String suiteId, @NonNull String suiteSecret, @NonNull String suiteTicket)
        throws WeworkException {

        String url = wwGetSuiteToken;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("suite_id", suiteId);
        params.put("suite_secret", suiteSecret);
        params.put("suite_ticket", suiteTicket);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<SuiteToken> response = restTemplate.postForEntity(url, request, SuiteToken.class);

        SuiteToken result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取预授权码
     * 
     * @param suiteAccessToken 第三方应用access_token,最长为512字节
     * @return
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90601
     */
    @GenIgnore
    public PreAuthCode getPreAuthCode(@NonNull String suiteAccessToken) throws WeworkException {

        String url = MessageFormat.format(wwGetPreAuthCode, suiteAccessToken);

        ResponseEntity<PreAuthCode> response = restTemplate.getForEntity(url, PreAuthCode.class);

        PreAuthCode result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
    
    /**
     * 设置授权配置
     * 
     * @param suiteAccessToken
     * @param preAuthCode 预授权码
     * @param authType 授权类型：0 正式授权， 1 测试授权。 默认值为0。
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90602
     */
    @GenIgnore
    public void setSessionInfo(@NonNull String suiteAccessToken, @NonNull String preAuthCode, @GenNullable SuiteAuthType authType) throws WeworkException  {
        
        String url = MessageFormat.format(wwSetSessionInfo, suiteAccessToken);
        
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        
        Map<String, Object> sessionInfo = Maps.newHashMap();
        if (authType != null) {
            sessionInfo.put("auth_type", authType.getValue());
        }
        
        Map<String, Object> params = Maps.newHashMap();
        params.put("pre_auth_code", preAuthCode);
        params.put("session_info", sessionInfo);
        
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        
        ErrorCode result = response.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 设置授权配置
     * 
     * @param suiteAccessToken
     * @param preAuthCode 预授权码
     * @param authType 授权类型：0 正式授权， 1 测试授权。 默认值为0。
     * @param appids
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90602
     */
    @GenIgnore
    public void setSessionInfo(@NonNull String suiteAccessToken, @NonNull String preAuthCode, @GenNullable SuiteAuthType authType, @GenNullable String ... appids) throws WeworkException  {
        
        String url = MessageFormat.format(wwSetSessionInfo, suiteAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> sessionInfo = Maps.newHashMap();
        if (authType != null) {
            sessionInfo.put("auth_type", authType.getValue());
        }
        if (appids != null) {
            sessionInfo.put("appid", appids);
        }
        
        Map<String, Object> params = Maps.newHashMap();
        params.put("pre_auth_code", preAuthCode);
        params.put("session_info", sessionInfo);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        ErrorCode result = response.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 获取企业永久授权码
     * 
     * @param suiteAccessToken
     * @param authCode 临时授权码，会在授权成功时附加在redirect_uri中跳转回第三方服务商网站，或通过授权成功通知回调推送给服务商。长度为64至512个字节
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90603
     */
    @GenIgnore
    public PermanentCode getPermanentCode(@NonNull String suiteAccessToken, @NonNull String authCode) throws WeworkException {

        String url = MessageFormat.format(wwGetPermanentCode, suiteAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("auth_code", authCode);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<PermanentCode> response = restTemplate.postForEntity(url, request, PermanentCode.class);

        PermanentCode result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取企业授权信息
     * 
     * @param suiteAccessToken
     * @param autoCorpId
     * @param permanentCode
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90604
     */
    @GenIgnore
    public AuthInfo getAuthInfo(@NonNull String suiteAccessToken, @NonNull String autoCorpId, @NonNull String permanentCode) throws WeworkException {
        String url = MessageFormat.format(wwGetAuthInfo, suiteAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("auth_corpid", autoCorpId);
        params.put("permanent_code", permanentCode);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AuthInfo> response = restTemplate.postForEntity(url, request, AuthInfo.class);

        AuthInfo result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取企业凭证
     * 
     * @param suiteAccessToken
     * @param authCorpId 授权方corpid
     * @param permanentCode 永久授权码，通过getPermanentCode获取
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90605
     */
    @GenIgnore
    public CorpToken getCorpToken(@NonNull String suiteAccessToken, @NonNull String authCorpId, @NonNull String permanentCode) throws WeworkException {

        String url = MessageFormat.format(wwGetCorpToken, suiteAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("auth_corpid", authCorpId);
        params.put("permanent_code", permanentCode);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<CorpToken> response = restTemplate.postForEntity(url, request, CorpToken.class);

        CorpToken result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取应用的管理员列表
     * 
     * @param suiteAccessToken
     * @param autoCorpId
     * @param agentId
     * @return
     * @throws WeworkException
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90606
     */
    @GenIgnore
    public AdminList getAdminList(@NonNull String suiteAccessToken, @NonNull String autoCorpId, @NonNull String agentId) throws WeworkException {
        String url = MessageFormat.format(wwGetAdminList, suiteAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("auth_corpid", autoCorpId);
        params.put("agentid", agentId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AdminList> response = restTemplate.postForEntity(url, request, AdminList.class);

        AdminList result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
}
