/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.contact.department.Department;
import cn.kinyun.wework.sdk.entity.contact.department.DepartmentList;
import cn.kinyun.wework.sdk.entity.contact.department.DepartmentSimpleList;
import cn.kinyun.wework.sdk.entity.contact.department.IdResp;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;

import lombok.NonNull;

/**
 * 通讯录API
 * 
 * @title UserApi
 * @author yanmaoyuan
 * @date 2021年3月13日
 * @version 1.0
 */
@Component
public class DepartmentApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.department.create}")
    private String createDepartmentUrl;

    @Value("${qyapi.department.update}")
    private String updateDepartmentUrl;

    @Value("${qyapi.department.delete}")
    private String delDepartmentUrl;

    @Value("${qyapi.department.list}")
    private String listDepartmentUrl;

    @Value("${qyapi.department.simplelist}")
    private String simpleListUrl;

    /**
     * 创建部门
     * 
     * @param accessToken
     * @param department 创建部门的参数
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90205
     */
    public Integer create(@NonNull String accessToken, @NonNull Department department) throws WeworkException {
        String url = MessageFormat.format(createDepartmentUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<Department> httpEntity = new HttpEntity<>(department, headers);
        ResponseEntity<IdResp> forEntity = restTemplate.postForEntity(url, httpEntity, IdResp.class);

        IdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getId();
    }

    /**
     * 更新部门
     * 
     * @param accessToken
     * @param department 更新部门的参数
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90206
     */
    public void update(@NonNull String accessToken, @NonNull Department department) throws WeworkException {
        String url = MessageFormat.format(updateDepartmentUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<Department> httpEntity = new HttpEntity<>(department, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * @param accessToken
     * @param id 删除的部门id
     * @desc 删除部门
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90207
     */
    public void delete(@NonNull String accessToken, @NonNull Integer id) throws WeworkException {
        String url = MessageFormat.format(delDepartmentUrl, accessToken, String.valueOf(id));

        ResponseEntity<ErrorCode> forEntity = restTemplate.getForEntity(url, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 查询部门列表
     *
     * @param accessToken
     * @param id 部门id。获取指定部门及其下的子部门（以及及子部门的子部门等等，递归）。 如果不填，默认获取全量组织架构
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90208
     */
    public List<Department> list(@NonNull String accessToken, @GenNullable Integer id) throws WeworkException {
        String url = MessageFormat.format(listDepartmentUrl, accessToken);

        if (id != null && id >= 1) {
            url = url + "&id=" + id;
        }

        ResponseEntity<DepartmentList> forEntity = restTemplate.getForEntity(url, DepartmentList.class);

        DepartmentList result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getDepartmentList();
    }

    /**
     * 获取子部门ID列表
     *
     * @param accessToken
     * @param id 部门id。获取指定部门及其下的子部门（以及及子部门的子部门等等，递归）。 如果不填，默认获取全量组织架构
     * @link https://developer.work.weixin.qq.com/document/path/95350
     */
    public List<Department> simpleList(@NonNull String accessToken, @GenNullable Integer id) throws WeworkException {
        String url = MessageFormat.format(simpleListUrl, accessToken);

        if (id != null && id >= 1) {
            url = url + "&id=" + id;
        }

        ResponseEntity<DepartmentSimpleList> forEntity = restTemplate.getForEntity(url, DepartmentSimpleList.class);

        DepartmentSimpleList result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getDepartmentList();
    }

}
