/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api.chat;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.entity.chat.ChatData;
import cn.kinyun.wework.sdk.entity.chat.ChatMsg;
import cn.kinyun.wework.sdk.entity.chat.EncryptChatData;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import cn.kinyun.wework.sdk.exception.FinanceSdkException;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.tencent.wework.Finance;

import lombok.extern.slf4j.Slf4j;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.Security;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Logger;

import javax.crypto.Cipher;

import lombok.NonNull;

/**
 * @title ChatApi
 * @author yanmaoyuan
 * @date 2021年3月18日
 * @version 1.0
 */
@Slf4j
@GenIgnore
public class ChatApi {

    static final ObjectMapper OBJ_MAPPER = new ObjectMapper();

    static {
        // BouncyCastle
        Security.addProvider(new BouncyCastleProvider());
        log.info("add BouncyCastle Provider");

        // 初始化json反序列化
        OBJ_MAPPER.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
        OBJ_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        OBJ_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * NewSdk返回的sdk指针
     */
    private volatile long sdk;

    /**
     * 调用企业的企业id，例如：wwd08c8exxxx5ab44d，可以在企业微信管理端--我的企业--企业信息查看
     */
    private String corpId;

    /**
     * 聊天内容存档的Secret，可以在企业微信管理端--管理工具--聊天内容存档查看
     */
    private String secret;

    private String proxy;

    private String passwd;

    private int timeout = 5;

    /**
     * 以版本号为索引，记录不同版本的RSA私钥。
     */
    private final Map<String, PrivateKey> privateKeyMap = new HashMap<>();

    /**
     * 
     * @param corpId 调用企业的企业id，例如：wwd08c8exxxx5ab44d，可以在企业微信管理端--我的企业--企业信息查看
     * @param secret 聊天内容存档的Secret，可以在企业微信管理端--管理工具--聊天内容存档查看
     * @throws Exception
     */
    public ChatApi(@NonNull String corpId, @NonNull String secret) {
        this.corpId = corpId;
        this.secret = secret;

        initSdk();
    }

    /**
     * 添加RSA私钥
     * 
     * @param version 密钥的版本号
     * @param privateKey RSA私钥
     * @throws Exception
     */
    public void addPrivateKey(@NonNull String version, @NonNull String privateKey) throws IOException {
        PrivateKey key = getPrivateKey(privateKey);
        privateKeyMap.put(version, key);
    }

    /**
     * 初始化SDK，若初始化失败将抛出异常。
     * 
     * @throws FinanceSdkException
     */
    public void initSdk() throws FinanceSdkException {
        if (sdk != 0L) {
            throw new IllegalStateException("sdk is already initialized.");
        }

        // 使用sdk前需要初始化，初始化成功后的sdk可以一直使用。
        // 如需并发调用sdk，建议每个线程持有一个sdk实例。
        // 初始化时请填入自己企业的corpid与secrectkey。
        long ptrSdk = Finance.NewSdk();
    
        long ret = Finance.Init(ptrSdk, corpId, secret);
        if (ret != 0) {
            Finance.DestroySdk(ptrSdk);
            throw new FinanceSdkException(ret, "init sdk failed");
        }

        this.sdk = ptrSdk;
    }

    /**
     * 销毁SDK
     */
    public void destroySdk() {
        if (sdk == 0L) {
            throw new IllegalStateException("sdk is not initialized before destroy.");
        }

        Finance.DestroySdk(sdk);
        sdk = 0L;
    }

    /**
     * 获取企业ID
     * 
     * @return
     */
    public String getCorpId() {
        return corpId;
    }

    public void setCorpId(String corpId) {
        this.corpId = corpId;
    }

    public void setSecret(String secret) {
        this.secret = secret;
    }

    /**
     * 使用代理的请求，需要传入代理的链接。如：socks5://10.0.0.1:8081 或者 http://10.0.0.1:8081
     * @param proxy
     */
    public void setProxy(String proxy) {
        this.proxy = proxy;
    }

    /**
     * 代理账号密码，需要传入代理的账号密码。如 user_name:passwd_123
     * 
     * @param passwd
     */
    public void setPasswd(String passwd) {
        this.passwd = passwd;
    }

    /**
     * 超时时间，单位秒
     * @param timeout
     */
    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    /**
     * 拉取聊天记录
     * 
     * @param seq 消息的seq值，标识消息的序号。再次拉取需要带上上次回包中最大的seq。
     * @param limit 一次拉取的消息条数，最大值1000条，超过1000条会返回错误。
     * @throws Exception
     * @return
     */
    public ChatData getChatdata(Long seq, Integer limit) throws WeworkException {
        if (sdk == 0L) {
            throw new IllegalStateException("sdk is not initialized before destroy.");
        }

        // 每次使用GetChatData拉取存档前需要调用NewSlice获取一个slice，在使用完slice中数据后，还需要调用FreeSlice释放。
        long slice = Finance.NewSlice();

        long ret = Finance.GetChatData(sdk, seq, limit, proxy, passwd, timeout, slice);
        if (ret != 0) {
            Finance.FreeSlice(slice);
            throw new FinanceSdkException(ret, "获取聊天数据失败");
        }
        String content = Finance.GetContentFromSlice(slice);

        // 释放内存
        Finance.FreeSlice(slice);

        // 反序列化为json
        ChatData chatdata = null;
        try {
            chatdata = OBJ_MAPPER.readValue(content, ChatData.class);
        } catch (IOException e) {
            e.printStackTrace();
            log.error("解析聊天数据失败:{}", content, e);
            throw new FinanceSdkException(ret, "解析聊天数据失败");
        }

        WeworkException.isSuccess(chatdata);
        return chatdata;
    }

    public File getMediadata(String sdkfileid, String savefile) {
        if (sdk == 0L) {
            throw new IllegalStateException("sdk is not initialized before destroy.");
        }
        
        long ret = 0;

        // 生成临时文件目录
        File dirFolder = new File(System.getProperty("java.io.tmpdir"), "WeWorkFinanceSdk");
        if (!dirFolder.exists()) {
            dirFolder.mkdirs();// NOSONAR
        }

        // 临时文件
        String tmpName = UUID.randomUUID().toString() + ".tmp";
        File tmpFile = new File(dirFolder, tmpName);

        // 保存的文件
        File file = new File(savefile);

        // 媒体文件每次拉取的最大size为512k，因此超过512k的文件需要分片拉取。若该文件未拉取完整，sdk的IsMediaDataFinish接口会返回0，同时通过GetOutIndexBuf接口返回下次拉取需要传入GetMediaData的indexbuf。
        // indexbuf一般格式如右侧所示，”Range:bytes=524288-1048575“，表示这次拉取的是从524288到1048575的分片。单个文件首次拉取填写的indexbuf为空字符串，拉取后续分片时直接填入上次返回的indexbuf即可。
        String indexbuf = "";
        while (true) {
            // 每次使用GetMediaData拉取存档前需要调用NewMediaData获取一个media_data，在使用完media_data中数据后，还需要调用FreeMediaData释放。
            long ptrMediaData = Finance.NewMediaData();
            ret = Finance.GetMediaData(sdk, indexbuf, sdkfileid, proxy, passwd, timeout, ptrMediaData);
            if (ret != 0) {
                Finance.FreeMediaData(ptrMediaData);
                throw new FinanceSdkException(ret, "get media data failed");
            }

            int indexLen = Finance.GetIndexLen(ptrMediaData);
            int dataLen = Finance.GetDataLen(ptrMediaData);
            int isFinish = Finance.IsMediaDataFinish(ptrMediaData);

            log.info("getmediadata outindex len:{}, data_len:{}, is_finish:{}", indexLen, dataLen, isFinish);

            try(FileOutputStream outputStream = new FileOutputStream(tmpFile, true)) {
                // 大于512k的文件会分片拉取，此处需要使用追加写，避免后面的分片覆盖之前的数据。
                outputStream.write(Finance.GetData(ptrMediaData));
            } catch (Exception e) {
                e.printStackTrace();
            }

            if (isFinish == 1) {
                // 已经拉取完成最后一个分片
                Finance.FreeMediaData(ptrMediaData);
                break;
            } else {
                // 获取下次拉取需要使用的indexbuf
                indexbuf = Finance.GetOutIndexBuf(ptrMediaData);
                Finance.FreeMediaData(ptrMediaData);
            }
        }

        try {
            Files.copy(tmpFile.toPath(), file.toPath(), StandardCopyOption.REPLACE_EXISTING);
            tmpFile.delete();//NOSONAR
        } catch (IOException ex) {
            ex.printStackTrace();
            log.error("delete file failed.", ex);
        }

        return file;
    }

    public List<ChatMsg> decryptAll(List<EncryptChatData> list) throws FinanceSdkException, GeneralSecurityException {
        if (sdk == 0L) {
            throw new IllegalStateException("sdk is not initialized before destroy.");
        }

        List<ChatMsg> results = new ArrayList<>(list.size());
        for (EncryptChatData data : list) {
            results.add(decrypt(data));
        }

        return results;
    }

    public ChatMsg decrypt(EncryptChatData encryptChatData) throws FinanceSdkException, GeneralSecurityException {
        if (sdk == 0L) {
            throw new IllegalStateException("sdk is not initialized before destroy.");
        }

        String publickeyVer = encryptChatData.getPublickeyVer();
        String encryptRandomKey = encryptChatData.getEncryptRandomKey();
        String encryptChatMsg = encryptChatData.getEncryptChatMsg();

        // 使用publickey_ver指定版本的私钥
        PrivateKey prikey = privateKeyMap.get(publickeyVer);
        // 检查publicVer对应的私钥是否存在
        if (prikey == null) {
            throw new NullPointerException("RSA Private Key not found with version:" + publickeyVer);
        }

        // 对每条消息的encrypt_random_key内容进行Base64解码
        byte[] randomKeyBytes = Base64.getDecoder().decode(encryptRandomKey);

        // 使用RSA PKCS1算法对randomKey进行解密，得到解密内容
        Cipher rsa = Cipher.getInstance("RSA/ECB/PKCS1Padding", BouncyCastleProvider.PROVIDER_NAME);//NOSONAR
        rsa.init(Cipher.DECRYPT_MODE, prikey);
        byte[] encryptKeyBytes = rsa.doFinal(randomKeyBytes);
        String encryptKey = new String(encryptKeyBytes, StandardCharsets.UTF_8);

        // 每次使用DecryptData解密会话存档前需要调用NewSlice获取一个slice，在使用完slice中数据后，还需要调用FreeSlice释放。
        long msg = Finance.NewSlice();
        long ret = Finance.DecryptData(sdk, encryptKey, encryptChatMsg, msg);
        if (ret != 0) {
            Finance.FreeSlice(msg);
            throw new FinanceSdkException(ret, "decrypt data failed");
        }

        String content = Finance.GetContentFromSlice(msg);
        Finance.FreeSlice(msg);

        ChatMsg chatMsg;
        try {
            chatMsg = OBJ_MAPPER.readValue(content, ChatMsg.class);
        } catch (IOException e) {
            e.printStackTrace();
            log.error("解析聊天数据失败:{}", content, e);
            throw new FinanceSdkException(ret, "解析聊天数据失败");
        }

        return chatMsg;
    }

    /**
     * 解析RSA私钥，转换为PrivateKey对象。
     * 
     * @param privateKey
     * @return
     * @throws Exception
     */
    private PrivateKey getPrivateKey(String privateKey) throws IOException {
        try (PEMParser parser = new PEMParser(new StringReader(privateKey))) {
            Object obj = parser.readObject();
            
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter();
            converter.setProvider(BouncyCastleProvider.PROVIDER_NAME);

            if (obj instanceof PEMKeyPair) {// RSA PRIVATE KEY
                PEMKeyPair pair = (PEMKeyPair) obj;

                PrivateKeyInfo privateKeyInfo = pair.getPrivateKeyInfo();
                return converter.getPrivateKey(privateKeyInfo);
            } else if (obj instanceof PrivateKeyInfo) {// PRIVATE KEY

                PrivateKeyInfo privateKeyInfo = (PrivateKeyInfo) obj;
                return converter.getPrivateKey(privateKeyInfo);
            } else {
                log.warn("read private key failed:{}", obj);
            }
        }

        return null;
    }

    public static ChatApiBuilder builder() {
        return new ChatApiBuilder();
    }
}
