package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.entity.material.GetMaterialResp;
import cn.kinyun.wework.sdk.entity.material.UploadAttachmentResult;
import cn.kinyun.wework.sdk.entity.material.UploadImageMaterialResult;
import cn.kinyun.wework.sdk.entity.material.UploadMaterialResult;
import cn.kinyun.wework.sdk.exception.WeworkException;
import cn.kinyun.wework.sdk.utils.JacksonUtils;

import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Arrays;

import lombok.NonNull;

/**
 * 素材管理api
 */
@GenIgnore
@Component
public class WwMaterialApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 上传临时素材
     */
    @Value("${qyapi.material.manager.upload}")
    private String materialUploadUrl;

    /**
     * 上传图片
     */
    @Value("${qyapi.material.manager.uploadimage}")
    private String materialUploadImageUrl;

    /**
     * 获取临时素材
     */
    @Value("${qyapi.material.manager.gettempmaterial}")
    private String getTempMaterialUrl;

    /**
     * 获取高清语音素材
     */
    @Value("${qyapi.material.manager.getvoicematerial}")
    private String getVoiceMaterialUrl;

    /**
     * 上传附件资源
     */
    @Value("${qyapi.material.manager.uploadattachment}")
    private String uploadAttachmentUrl;

    /**
     * 上传临时素材
     * 
     * @param accessToken
     * @param type 素材类型 必须
     * @param file 上传的素材文件 必须
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90389
     */
    public UploadMaterialResult uploadMaterial(@NonNull String accessToken, @NonNull String type, @NonNull File file) throws IOException, WeworkException {
        String url = MessageFormat.format(materialUploadUrl, accessToken, type);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        param.add("filelength", resource.contentLength());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(param, headers);

        ResponseEntity<UploadMaterialResult> forEntity = restTemplate.postForEntity(url, httpEntity, UploadMaterialResult.class);

        UploadMaterialResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 上传图片
     * 
     * @param accessToken
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90392
     */
    public UploadImageMaterialResult uploadImageMaterial(@NonNull String accessToken, @NonNull File file) throws IOException, WeworkException {
        String url = MessageFormat.format(materialUploadImageUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        param.add("filelength", resource.contentLength());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(param, headers);

        ResponseEntity<UploadImageMaterialResult> forEntity = restTemplate.postForEntity(url, httpEntity, UploadImageMaterialResult.class);

        UploadImageMaterialResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param mediaId  素材id
     * @param filePath 下载文件目录路径
     * @desc 获取素材临时文件
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90390
     */
    public GetMaterialResp getMaterial(@NonNull String accessToken, @NonNull String mediaId, @NonNull String filePath) throws WeworkException {
        String url = MessageFormat.format(getTempMaterialUrl, accessToken, mediaId);

        RequestCallback requestCallback = request -> request.getHeaders()
                .setAccept(Arrays.asList(MediaType.APPLICATION_OCTET_STREAM, MediaType.ALL));

        GetMaterialResp result = new GetMaterialResp();

        restTemplate.execute(url, HttpMethod.GET, requestCallback, clientHttpResponse -> {
            HttpHeaders headers = clientHttpResponse.getHeaders();
            InputStream body = clientHttpResponse.getBody();

            MediaType contentType = headers.getContentType();
            if (contentType != null && (contentType.getType().equals(MediaType.TEXT_PLAIN.getType())
                    || contentType.getType().equals(MediaType.APPLICATION_JSON.getType()))) {
                String textString = IOUtils.toString(body, StandardCharsets.UTF_8);
                GetMaterialResp resp = JacksonUtils.readValue(textString, GetMaterialResp.class);
                result.setErrCode(resp.getErrCode());
                result.setErrMsg(resp.getErrMsg());
            } else if (contentType != null) {
                //得到的是文件流
                ContentDisposition contentDisposition = headers.getContentDisposition();
                if ("attachment".equals(contentDisposition.getType())) {
                    String fileName = contentDisposition.getFilename();
                    StringBuilder mediaPath = new StringBuilder(filePath);
                    if (filePath.endsWith(File.separator)) {
                        mediaPath.append(fileName);
                    } else {
                        mediaPath.append(File.separator);
                        mediaPath.append(fileName);
                    }
                    FileOutputStream fileOutputStream = new FileOutputStream(mediaPath.toString());

                    IOUtils.copy(body, fileOutputStream);
                    String type = contentType.getType().split("/")[0];

                    result.setFileOutputStream(fileOutputStream);
                    result.setType(type);
                    result.setErrCode(0);
                }
            }
            return null;
        });
        WeworkException.isSuccess(result);
        return result;
    }

    /***
     * @param accessToken
     * @param mediaId 素材id
     * @param filePath 文件下载后存放的目录路径
     * @desc 获取高清语音素材
     * @link https://open.work.weixin.qq.com/api/doc/90001/90143/90391
     */
    public GetMaterialResp getVoiceMaterial(@NonNull String accessToken, @NonNull String mediaId, @NonNull String filePath) throws WeworkException {
        String url = MessageFormat.format(getVoiceMaterialUrl, accessToken, mediaId);

        RequestCallback requestCallback = request -> request.getHeaders()
                .setAccept(Arrays.asList(MediaType.APPLICATION_OCTET_STREAM, MediaType.ALL));

        GetMaterialResp result = new GetMaterialResp();

        restTemplate.execute(url, HttpMethod.GET, requestCallback, clientHttpResponse -> {
            HttpHeaders headers = clientHttpResponse.getHeaders();
            InputStream body = clientHttpResponse.getBody();

            MediaType contentType = headers.getContentType();
            //出错返回的是json数据
            if (contentType != null && (contentType.getType().equals(MediaType.TEXT_PLAIN.getType())
                    || contentType.getType().equals(MediaType.APPLICATION_JSON.getType()))) {
                String textString = IOUtils.toString(body, StandardCharsets.UTF_8);
                GetMaterialResp resp = JacksonUtils.readValue(textString, GetMaterialResp.class);
                result.setErrCode(resp.getErrCode());
                result.setErrMsg(resp.getErrMsg());
            } else if (contentType != null) {
                //得到的是文件流
                ContentDisposition contentDisposition = headers.getContentDisposition();
                if ("attachment".equals(contentDisposition.getType())) {
                    String fileName = contentDisposition.getFilename();
                    StringBuilder mediaPath = new StringBuilder(filePath);
                    if (filePath.endsWith(File.separator)) {
                        mediaPath.append(fileName);
                    } else {
                        mediaPath.append(File.separator);
                        mediaPath.append(fileName);
                    }
                    FileOutputStream fileOutputStream = new FileOutputStream(mediaPath.toString());

                    IOUtils.copy(body, fileOutputStream);
                    String type = contentType.getType().split("/")[0];

                    result.setFileOutputStream(fileOutputStream);
                    result.setType(type);
                    result.setErrCode(0);
                }
            }
            return null;
        });
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 上传附件资源
     *
     * @param accessToken
     * @param mediaType 媒体文件类型 必须 媒体文件类型，分别有图片（image）、视频（video）、普通文件（file）
     * @param attachmentType 附件类型 必须 附件类型，不同的附件类型用于不同的场景。1：朋友圈；2:商品图册
     * @param file 上传的素材文件 必须
     * @link <a href="https://developer.work.weixin.qq.com/document/path/95098">...</a>
     */
    public UploadAttachmentResult uploadAttachment(@NonNull String accessToken, @NonNull String mediaType, @NonNull Integer attachmentType, @NonNull File file) throws IOException, WeworkException {
        String url = MessageFormat.format(uploadAttachmentUrl, accessToken, mediaType, attachmentType);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);
        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        param.add("filelength", resource.contentLength());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param, headers);
        ResponseEntity<UploadAttachmentResult> forEntity = restTemplate.postForEntity(url, httpEntity, UploadAttachmentResult.class);
        UploadAttachmentResult result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
}
