/**
 * kuaikeguanjia.com Inc.
 * Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.annotation.GenLock;
import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.external.transfer.TransferCustomerParams;
import cn.kinyun.wework.sdk.entity.external.transfer.TransferCustomerResp;
import cn.kinyun.wework.sdk.entity.external.transfer.TransferGroupChatResp;
import cn.kinyun.wework.sdk.entity.external.transfer.TransferResult;
import cn.kinyun.wework.sdk.entity.external.transfer.UnassignedListResp;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import lombok.NonNull;

/**
 * 在职/离职继承
 * 
 * @link 在职继承 https://open.work.weixin.qq.com/api/doc/90000/90135/92125
 * @link 离职继承 https://open.work.weixin.qq.com/api/doc/90000/90135/92124
 * @title TransferApi
 * @author yanmaoyuan
 * @date 2021年4月28日
 * @version 1.0
 */
@Component
public class TransferApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 分配在职成员客户
     */
    @Value("${qyapi.externalcontact.transfer_customer}")
    private String transferCustomerUrl;

    /**
     * 查询客户接替状态
     */
    @Value("${qyapi.externalcontact.transfer_result}")
    private String transferResultUrl;

    /**
     * 获取待分配的离职成员列表
     */
    @Value("${qyapi.externalcontact.get_unassigned_list}")
    private String unassignedListUrl;

    /**
     * 分配离职成员的客户
     */
    @Value("${qyapi.externalcontact.resigned.transfer_customer}")
    private String transferCustomerOfDimissionUrl;

    /**
     * 查询离职成员客户接替状态
     */
    @Value("${qyapi.externalcontact.resigned.transfer_result}")
    private String transferResultOfDimissionUrl;

    /**
     * 分配离职成员的客户群
     */
    @Value("${qyapi.externalcontact.groupchat.transfer}")
    private String transferGroupChatUrl;


    /**
     * @param accessToken
     * @param params
     * @desc 分配在职成员的客户, 企业可通过此接口，转接在职成员的客户给其他成员。
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/94083
     */
    @GenAsync
    public TransferCustomerResp transferCustomer(@NonNull String accessToken, @NonNull TransferCustomerParams params) throws WeworkException {
        String url = MessageFormat.format(transferCustomerUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<TransferCustomerParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<TransferCustomerResp> forEntity = restTemplate.postForEntity(url, httpEntity, TransferCustomerResp.class);

        TransferCustomerResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param handoverUserId 原添加成员的userid
     * @param takeoverUserId 接替成员的userid
     * @param cursor 分页查询的cursor，每个分页返回的数据不会超过1000条；不填或为空表示获取第一个分页；
     * @desc 查询客户接替状态, 企业和第三方可通过此接口查询在职成员的客户转接情况
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/94088
     */
    public TransferResult getTransferResult(@NonNull String accessToken, @NonNull String handoverUserId,
                                            @NonNull String takeoverUserId, @GenNullable String cursor) throws WeworkException {
        String url = MessageFormat.format(transferResultUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("handover_userid", handoverUserId);
        params.put("takeover_userid", takeoverUserId);
        params.put("cursor", cursor);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<TransferResult> forEntity = restTemplate.postForEntity(url, httpEntity, TransferResult.class);

        TransferResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param pageId 开始页的id 分页查询，要查询页号，从0开始
     * @param pageSize
     * @param cursor
     * @desc 获取待分配的离职成员列表
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92124
     */
    @GenLock
    public UnassignedListResp getUnassignedList(@NonNull String accessToken, @GenNullable Integer pageId,
                                                @GenNullable Integer pageSize, @GenNullable String cursor) throws WeworkException {
        String url = MessageFormat.format(unassignedListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("page_id", pageId);
        params.put("page_size", pageSize);
        params.put("cursor", cursor);


        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<UnassignedListResp> forEntity = restTemplate.postForEntity(url, httpEntity, UnassignedListResp.class);

        UnassignedListResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param params  离职转交客户的参数
     * @desc 分配离职成员的客户  和在职转交的参数基本一样 返回结果也一样
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/94081
     */
    @GenAsync
    public TransferCustomerResp dimissionTransferCustomer(@NonNull String accessToken, @NonNull TransferCustomerParams params) throws WeworkException {
        String url = MessageFormat.format(transferCustomerOfDimissionUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<TransferCustomerParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<TransferCustomerResp> forEntity = restTemplate.postForEntity(url, httpEntity, TransferCustomerResp.class);

        TransferCustomerResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }


    /**
     * @param accessToken
     * @param handoverUserId 原添加成员的userid
     * @param takeoverUserId 接替成员的userid
     * @param cursor 分页查询的cursor，每个分页返回的数据不会超过1000条；不填或为空表示获取第一个分页；
     * @desc 查询离职成员客户接替状态, 企业和第三方可通过此接口查询离职成员的客户分配情况。
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/94082
     */
    public TransferResult getTransferResultOfDimission(@NonNull String accessToken, @NonNull String handoverUserId,
                                            @NonNull String takeoverUserId, @GenNullable String cursor) throws WeworkException {
        String url = MessageFormat.format(transferResultOfDimissionUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("handover_userid", handoverUserId);
        params.put("takeover_userid", takeoverUserId);
        params.put("cursor", cursor);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<TransferResult> forEntity = restTemplate.postForEntity(url, httpEntity, TransferResult.class);

        TransferResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param chatIds 需要转群主的客户群ID列表。取值范围： 1 ~ 100
     * @param newOwner 新群主ID
     * @desc 分配离职成员的客户群，企业可通过此接口，将已离职成员为群主的群，分配给另一个客服成员。
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92127
     */
    public TransferGroupChatResp transferGroupChat(@NonNull String accessToken, @NonNull Set<String> chatIds, @NonNull String newOwner) throws WeworkException {
        String url = MessageFormat.format(transferGroupChatUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("chat_id_list", chatIds);
        params.put("new_owner", newOwner);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<TransferGroupChatResp> forEntity = restTemplate.postForEntity(url, httpEntity, TransferGroupChatResp.class);

        TransferGroupChatResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }
}
