/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.annotation.GenLock;
import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.annotation.GenRetry;
import cn.kinyun.wework.sdk.entity.external.contact.*;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.exception.WeworkException;

import cn.kinyun.wework.sdk.utils.JacksonUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 客户联系API
 * 
 * @title ExternalContactApi
 * @author yanmaoyuan
 * @date 2021年3月13日
 * @version 1.0
 */
@Component
public class ExternalContactApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 获取客户列表url
     */
    @Value("${qyapi.externalcontact.list}")
    private String extContactListUrl;

    /**
     * 获取客户详情
     */
    @Value("${qyapi.externalcontact.get}")
    private String externalContactDetailUrl;

    /**
     * 批量获取客户详情信息 获取成员一页客户的详情列表
     */
    @Value("${qyapi.externalcontact.batch_get_by_user}")
    private String externalContactDetailListUrl;

    /**
     * 修改客户备注
     */
    @Value("${qyapi.externalcontact.remark}")
    private String externalContactRemarkUrl;

    /**
     * unionid转external_userid
     */
    @Value("${qyapi.externalcontact.unionid_to_external_userid}")
    private String unionidToExternalUserIdUrl;


    @Value("${qyapi.externalcontact.unionid_convert}")
    private String unionidConvertUrl;

    @Value("${qyapi.externalcontact.external_userid_get_pendingid}")
    private String userIdGetPendingIdUrl;

    /**
     * 获取客户(ID)列表
     * 
     * @param accessToken
     * @param userId 成员id
     * @link https://developer.work.weixin.qq.com/document/path/92113
     */
    public List<String> list(@NonNull String accessToken, @NonNull String userId) throws WeworkException {
        String url = MessageFormat.format(extContactListUrl, accessToken, userId);

        ResponseEntity<ExternalContactList> forEntity = restTemplate.getForEntity(url, ExternalContactList.class);

        ExternalContactList result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getExternalUserList();
    }

    /**
     * 获取客户详情
     * 
     * @param accessToken
     * @param externalUserId 客户id
     * @link https://developer.work.weixin.qq.com/document/path/92114
     */
    public ExternalContactResp getDetail(@NonNull String accessToken, @NonNull String externalUserId, @GenNullable String cursor)
        throws WeworkException {
        String url = MessageFormat.format(externalContactDetailUrl, accessToken, externalUserId);

        if (cursor != null) {
            url = url + "&cursor=" + cursor;
        }

        ResponseEntity<String> forEntity = restTemplate.getForEntity(url, String.class);
        String body = forEntity.getBody();

        ExternalContactResp result = null;
        try {
            result = JacksonUtils.readValue(body, ExternalContactResp.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 批量获取客户详情，分页查询
     * 
     * @param accessToken
     * @param userId 成员id
     * @param cursor 接口分页参数， 游标指定获取下一页
     * @param limit 页大小 默认是50
     * @link https://developer.work.weixin.qq.com/document/path/92994
     */
    public ExternalContactDetailList getDetailList(@NonNull String accessToken,
                                                   @NonNull String userId, @GenNullable String cursor, @GenNullable Integer limit) throws WeworkException {
        String url = MessageFormat.format(externalContactDetailListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("userid", userId);
        if (cursor != null) {
            params.put("cursor", cursor);
        }
        if (limit != null) {
            params.put("limit", limit);
        }
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<String> forEntity =
            restTemplate.postForEntity(url, httpEntity, String.class);
        String body = forEntity.getBody();
        ExternalContactDetailList result = null;
        try {
            result = JacksonUtils.readValue(body, ExternalContactDetailList.class);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 修改客户备注信息
     * 
     * @param accessToken
     * @param remark 备注的具体信息
     * @link https://developer.work.weixin.qq.com/document/path/92115
     */
    @GenAsync
    @GenRetry
    @GenLock("\"ExternalContactClient:remark:\" + corpId ")
    public void remark(@NonNull String accessToken, @NonNull ExternalContactRemark remark) throws WeworkException {
        // 注意：如果要把客户的备注手机号清空，需要传入空字符串""。如果直接传入null或者空集合，企微后台并不会修改手机号。
        if (remark.getRemarkMobiles() != null && remark.getRemarkMobiles().isEmpty()) {
            // 要和客户端约定，传入null表示不想改手机号，但是传入空集合表示要把手机号清空
            remark.setRemarkMobiles(Collections.singletonList(""));
        }
        String url = MessageFormat.format(externalContactRemarkUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<ExternalContactRemark> httpEntity = new HttpEntity<>(remark, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 企业主体unionid转换为第三方external_userid
     *
     * @link https://developer.work.weixin.qq.com/document/path/93274
     * @param accessToken 令牌
     * @param unionId unionId
     * @param openId openId
     */
    @GenLock
    public String unionIdToExternalUserId(@NonNull String accessToken, @NonNull String unionId, @NonNull String openId) throws WeworkException {
        String url = MessageFormat.format(unionidToExternalUserIdUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("unionid", unionId);
        params.put("openid", openId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ExternalUserIdResp> forEntity =
                restTemplate.postForEntity(url, httpEntity, ExternalUserIdResp.class);

        ExternalUserIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getExternalUserId();
    }

    /**
     * uionid 转external_userid 或者 pendingId
     * @param accessToken
     * @param unionId
     * @param openId
     * @return
     * @throws WeworkException
     */
    @GenLock
    public ExternalUserIdResp unionIdConvert(@NonNull String accessToken, @NonNull String unionId, @NonNull String openId) throws WeworkException {
        String url = MessageFormat.format(unionidConvertUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("unionid", unionId);
        params.put("openid", openId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ExternalUserIdResp> forEntity =
                restTemplate.postForEntity(url, httpEntity, ExternalUserIdResp.class);

        ExternalUserIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    @GenLock
    public ExternalPendingIdResp userIdGetPendingId(@NonNull String accessToken, @NonNull List<String> externalUserIds) throws WeworkException {
        String url = MessageFormat.format(unionidConvertUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("external_userid", externalUserIds);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ExternalPendingIdResp> forEntity =
                restTemplate.postForEntity(url, httpEntity, ExternalPendingIdResp.class);

        ExternalPendingIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }
}
