/**
 * kuaikeguanjia.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.external.groupchat.ChatDetailInfo;
import cn.kinyun.wework.sdk.entity.external.groupchat.ExternalGroupChatDetailResp;
import cn.kinyun.wework.sdk.entity.external.groupchat.ExternalGroupChatListResp;
import cn.kinyun.wework.sdk.enums.ChatStatusFilter;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 客户群API
 * 
 * @title ExternalChatApi
 * @author yanmaoyuan
 * @date 2021年3月13日
 * @version 1.0
 */
@Component
public class ExternalChatApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 获取客户群列表
     */
    @Value("${qyapi.externalcontact.groupchat_list}")
    private String groupChatListUrl;

    /**
     * 获取客户群详情
     */
    @Value("${qyapi.externalcontact.groupchat_get}")
    private String groupChatDetailUrl;

    /**
     * 获取客户群列表
     * 
     * @param accessToken
     * @param statusFilter 客户群跟进状态过滤，默认为0。 0 - 所有列表(即不过滤) 1 - 离职待继承 2 - 离职继承中 3 - 离职继承完成
     * @param userIds 群主过滤，最多100个用户ID列表。如果不填，表示获取应用可见范围内全部群主的数据（但是不建议这么用，如果可见范围人数超过1000人，为了防止数据包过大，会报错 81017）
     * @param cursor 用于分页查询的游标，字符串类型，由上一次调用返回，首次调用不填
     * @param limit 分页，预期请求的数据量，取值范围 1 ~ 1000
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92120
     * <a href="https://developer.work.weixin.qq.com/document/path/92120">doc pash 变更</a>
     *
     */
    public ExternalGroupChatListResp list(@NonNull String accessToken, @GenNullable ChatStatusFilter statusFilter,
                                          @GenNullable List<String> userIds, @GenNullable String cursor, //
                                          @NonNull Integer limit) throws WeworkException {
        String url = MessageFormat.format(groupChatListUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        // 客户群跟进状态过滤。
        if (statusFilter != null) {
            params.put("status_filter", statusFilter.getValue());
        }
        // 群主过滤。用户ID列表。最多100个
        if (userIds != null && !userIds.isEmpty()) {
            Map<String, Object> ownerFilter = new HashMap<>();
            ownerFilter.put("userid_list", userIds);
            params.put("owner_filter", ownerFilter);
        }
        // 用于分页查询的游标，字符串类型，由上一次调用返回，首次调用不填
        if (cursor != null) {
            params.put("cursor", cursor);
        }
        // 分页，预期请求的数据量，取值范围 1 ~ 1000
        params.put("limit", limit);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ExternalGroupChatListResp> forEntity =
            restTemplate.postForEntity(url, httpEntity, ExternalGroupChatListResp.class);

        ExternalGroupChatListResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @param chatId 客户群id
     * @desc 获取客户群详情
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92122
     */
    public ChatDetailInfo detail(@NonNull String accessToken, @NonNull String chatId)
        throws WeworkException {
        String url = MessageFormat.format(groupChatDetailUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("chat_id", chatId);
        params.put("need_name", 1);// 是否需要返回群成员的名字
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ExternalGroupChatDetailResp> forEntity =
            restTemplate.postForEntity(url, httpEntity, ExternalGroupChatDetailResp.class);

        ExternalGroupChatDetailResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getGroupChat();
    }
}
