package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.external.customeracquisition.*;
import cn.kinyun.wework.sdk.exception.WeworkException;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

/**
 * 获客助手API
 *
 * 企业可通过此API创建获客链接，将链接分享给客户，客户点击链接后即可成为企业的客户。
 * @link https://developer.work.weixin.qq.com/document/path/97297
 * @link https://developer.work.weixin.qq.com/document/path/97298
 * @link https://developer.work.weixin.qq.com/document/path/97375
 * @author yanmaoyuan
 * @date 2023/5/6
 */
@Slf4j
@Component
public class CustomerAcquisitionApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.externalcontact.customer_acquisition.list_link}")
    private String listLinkUrl;// 获取获客链接列表
    @Value("${qyapi.externalcontact.customer_acquisition.get}")
    private String getLinkUrl;// 获取获客链接详情
    @Value("${qyapi.externalcontact.customer_acquisition.create_link}")
    private String createLinkUrl;// 创建获客链接
    @Value("${qyapi.externalcontact.customer_acquisition.update_link}")
    private String updateLinkUrl;// 编辑获客链接
    @Value("${qyapi.externalcontact.customer_acquisition.delete_link}")
    private String deleteLinkUrl;// 删除获客链接
    @Value("${qyapi.externalcontact.customer_acquisition.customer}")
    private String getLinkCustomerUrl;// 获取获客客户列表
    @Value("${qyapi.externalcontact.customer_acquisition.quota}")
    private String getQuotaUrl;// 查询剩余使用量

    /**
     * 获取获客链接列表
     *
     * 企业可通过此接口获取当前仍然有效的获客链接。
     * @param accessToken 调用接口凭证
     * @param limit 返回的最大记录数，整型，最大值100
     * @param cursor 用于分页查询的游标，字符串类型，由上一次调用返回，首次调用可不填
     * @return CustomerAcquisitionLinkListResp
     * @throws WeworkException 企微异常
     */
    public CustomerAcquisitionLinkList listLink(@NonNull String accessToken, Integer limit, String cursor) throws WeworkException {
        String url = MessageFormat.format(listLinkUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = new HashMap<>();
        if (limit != null) {
            params.put("limit", limit);
        }
        if (cursor != null) {
            params.put("cursor", cursor);
        }
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<CustomerAcquisitionLinkList> forEntity = restTemplate.postForEntity(url, httpEntity, CustomerAcquisitionLinkList.class);
        CustomerAcquisitionLinkList result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取获客链接详情
     *
     * 企业可通过此接口根据获客链接id获取链接配置详情。
     * @param accessToken 接口调用凭证
     * @param linkId 获客链接ID
     * @return 链接配置详情
     * @throws WeworkException 企微异常
     */
    public CustomerAcquisitionLinkDetail getLink(@NonNull String accessToken, @NonNull String linkId) throws WeworkException {
        String url = MessageFormat.format(getLinkUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = new HashMap<>();
        params.put("link_id", linkId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<CustomerAcquisitionLinkDetail> forEntity = restTemplate.postForEntity(url, httpEntity, CustomerAcquisitionLinkDetail.class);
        CustomerAcquisitionLinkDetail result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 创建获客链接
     *
     * 企业可通过此接口创建新的获客链接。
     * @param accessToken 接口调用凭证
     * @param linkReq 获客链接参数
     * @return 获客链接信息
     * @throws WeworkException 企微异常
     */
    public CustomerAcquisitionLinkResp createLink(@NonNull String accessToken, @NonNull CustomerAcquisitionLinkReq linkReq) throws WeworkException {
        String url = MessageFormat.format(createLinkUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CustomerAcquisitionLinkReq> httpEntity = new HttpEntity<>(linkReq, headers);
        ResponseEntity<CustomerAcquisitionLinkResp> forEntity = restTemplate.postForEntity(url, httpEntity, CustomerAcquisitionLinkResp.class);
        CustomerAcquisitionLinkResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 编辑获客链接
     *
     * 企业可通过此接口编辑获客链接，修改获客链接的关联范围或修改获客链接的名称。
     * @param accessToken 接口调用凭证
     * @param linkReq 获客链接参数
     * @throws WeworkException 企微异常
     */
    @GenAsync
    public void updateLink(@NonNull String accessToken, @NonNull CustomerAcquisitionLinkReq linkReq) throws WeworkException {
        String url = MessageFormat.format(updateLinkUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CustomerAcquisitionLinkReq> httpEntity = new HttpEntity<>(linkReq, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);
        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 删除获客链接
     *
     * 企业可通过此接口删除获客链接，删除后的获客链接将无法继续使用。
     * @param accessToken 接口调用凭证
     * @param linkId 获客链接ID
     * @throws WeworkException 企微异常
     */
    @GenAsync
    public void deleteLink(@NonNull String accessToken, @NonNull String linkId) throws WeworkException {
        String url = MessageFormat.format(deleteLinkUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = new HashMap<>();
        params.put("link_id", linkId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);
        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 获取获客客户列表
     *
     * 企业可通过此接口获取到由指定的获客链接添加的客户列表。
     * @link https://developer.work.weixin.qq.com/document/path/97298
     * @param accessToken 接口调用凭证
     * @param linkId 获客链接ID
     * @param limit 返回的最大记录数，整型，最大值1000
     * @param cursor 用于分页查询的游标，字符串类型，由上一次调用返回，首次调用可不填
     * @return 获客链接添加客户列表
     * @throws WeworkException 企微异常
     */
    public CustomerAcquisitionContactList getLinkCustomer(@NonNull String accessToken, @NonNull String linkId, Integer limit, String cursor) throws WeworkException {
        String url = MessageFormat.format(getLinkCustomerUrl, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = new HashMap<>();
        params.put("link_id", linkId);
        if (limit != null) {
            params.put("limit", limit);
        }
        if (cursor != null) {
            params.put("cursor", cursor);
        }
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<CustomerAcquisitionContactList> forEntity = restTemplate.postForEntity(url, httpEntity, CustomerAcquisitionContactList.class);
        CustomerAcquisitionContactList result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 查询剩余使用量
     *
     * 企业可通过此接口查询当前剩余的使用量。
     * @link https://developer.work.weixin.qq.com/document/path/97375
     * @param accessToken 接口调用凭证
     * @return 获客链接剩余使用量
     * @throws WeworkException 企微异常
     */
    public CustomerAcquisitionQuota getQuota(@NonNull String accessToken) throws WeworkException {
        String url = MessageFormat.format(getQuotaUrl, accessToken);
        ResponseEntity<CustomerAcquisitionQuota> forEntity = restTemplate.getForEntity(url, CustomerAcquisitionQuota.class);
        CustomerAcquisitionQuota result = forEntity.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
}