/**
 * kuaikeguanjia.com Inc.
 * Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api.external;

import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.external.contactway.ConfigIdResp;
import cn.kinyun.wework.sdk.entity.external.contactway.ContactWay;
import cn.kinyun.wework.sdk.entity.external.contactway.ContactWayResp;
import cn.kinyun.wework.sdk.entity.external.contactway.FollowerUserResp;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 企业服务人员管理
 * 
 * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92570
 * @link https://open.work.weixin.qq.com/api/doc/90000/90135/92572
 * @title ContactWayApi
 * @author yanmaoyuan
 * @date 2021年4月28日
 * @version 1.0
 */
@Component
public class ContactWayApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.externalcontact.get_follow_user_list}")
    private String getFollowUserListUrl;

    @Value("${qyapi.externalcontact.add_contact_way}")
    private String addContactWayUrl;

    @Value("${qyapi.externalcontact.get_contact_way}")
    private String getContactWayUrl;

    @Value("${qyapi.externalcontact.update_contact_way}")
    private String updateContactWayUrl;

    @Value("${qyapi.externalcontact.del_contact_way}")
    private String delContactWayUrl;

    /**
     * 获取配置了客户联系功能的成员列表
     * 
     * @param accessToken
     * @link https://work.weixin.qq.com/api/doc/90000/90135/92571
     */
    public List<String> getFollowUserList(@NonNull String accessToken) throws WeworkException {
        String url = MessageFormat.format(getFollowUserListUrl, accessToken);

        ResponseEntity<FollowerUserResp> forEntity = restTemplate.getForEntity(url, FollowerUserResp.class);

        FollowerUserResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getFollowUser();
    }

    /**
     * 获取企业已配置的「联系我」方式
     * 
     * @param accessToken
     * @param configId
     */
    public ContactWay get(@NonNull String accessToken, @NonNull String configId) throws WeworkException {
        String url = MessageFormat.format(getContactWayUrl, accessToken);
    
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
    
        Map<String, Object> params = new HashMap<>();
        params.put("config_id", configId);
    
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
    
        ResponseEntity<ContactWayResp> forEntity = restTemplate.postForEntity(url, httpEntity, ContactWayResp.class);
    
        ContactWayResp result = forEntity.getBody();
        WeworkException.isSuccess(result);
    
        return result.getContactWay();
    }

    /**
     * 配置客户联系「联系我」方式
     * 
     * @param accessToken
     * @param params
     * @link https://work.weixin.qq.com/api/doc/90000/90135/92572
     */
    @GenAsync
    public ConfigIdResp create(@NonNull String accessToken, @NonNull ContactWay params) throws WeworkException {
        String url = MessageFormat.format(addContactWayUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<ContactWay> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<ConfigIdResp> forEntity = restTemplate.postForEntity(url, httpEntity, ConfigIdResp.class);

        ConfigIdResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 更新企业已配置的「联系我」方式
     * 
     * @param accessToken
     * @param contactWay
     */
    @GenAsync
    public void update(@NonNull String accessToken, @NonNull ContactWay contactWay) throws WeworkException {
        String url = MessageFormat.format(updateContactWayUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<ContactWay> httpEntity = new HttpEntity<>(contactWay, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 删除企业已配置的「联系我」方式
     * 
     * @param accessToken
     * @param configId  配置id
     */
    @GenAsync
    public void delete(@NonNull String accessToken, @NonNull String configId) throws WeworkException {
        String url = MessageFormat.format(delContactWayUrl, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = new HashMap<>();
        params.put("config_id", configId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }
}
