/**
 * kuaikeguanjia.com Inc.
 * Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.contact.department.Department;
import cn.kinyun.wework.sdk.entity.contact.user.UserInfo;
import cn.kinyun.wework.sdk.entity.contact.user.UserListResp;
import cn.kinyun.wework.sdk.entity.contact.user.UserSimpleInfo;
import cn.kinyun.wework.sdk.entity.contact.user.UserSimpleListResp;
import cn.kinyun.wework.sdk.exception.WeworkException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 通讯录API
 * 
 * @title UserApi
 * @author yanmaoyuan
 * @date 2021年3月13日
 * @version 1.0
 */
@Component
public class UserApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.user.get}")
    private String wwUserGet;

    @Value("${qyapi.user.update}")
    private String wwUserUpdate;

    @Value("${qyapi.user.simplelist}")
    private String wwUserSimpleList;

    @Value("${qyapi.user.list}")
    private String wwUserList;

    @Value("${qyapi.user.create}")
    private String wwUserCreate;

    /**
     * @param accessToken 调用接口凭证
     * @param userId 成员UserID。对应管理端的帐号，企业内必须唯一。不区分大小写，长度为1~64个字节
     * @desc 读取成员信息
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90196
     */
    public UserInfo get(@NonNull String accessToken, @NonNull String userId) throws WeworkException {
        String url = MessageFormat.format(wwUserGet, accessToken, userId);

        ResponseEntity<UserInfo> forEntity = restTemplate.getForEntity(url, UserInfo.class);

        UserInfo userInfo = forEntity.getBody();
        WeworkException.isSuccess(userInfo);

        return userInfo;
    }

    /**
     * @param accessToken
     * @param departmentId 部门id
     * @param fetchChild 是否递归获取子部门下面的成员：1-递归获取，0-只获取本部门
     * @desc 获取部门成员
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90200
     */
    public List<UserSimpleInfo> simpleList(@NonNull String accessToken, @NonNull Integer departmentId, @GenNullable Integer fetchChild) throws WeworkException {
        if (fetchChild == null) {
            fetchChild = 0;
        }
        String url = MessageFormat.format(wwUserSimpleList, accessToken, String.valueOf(departmentId), fetchChild);

        ResponseEntity<UserSimpleListResp> forEntity = restTemplate.getForEntity(url, UserSimpleListResp.class);

        UserSimpleListResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getUserList();
    }

    /**
     * @param accessToken
     * @param departmentId 部门id
     * @param fetchChild 是否递归获取子部门下面的成员：1-递归获取，0-只获取本部门
     * @desc 获取部门成员详情
     * @link https://work.weixin.qq.com/api/doc/90000/90135/90201
     */
    public List<UserInfo> list(@NonNull String accessToken, @NonNull Integer departmentId, @GenNullable Integer fetchChild) throws WeworkException {
        if (fetchChild == null) {
            fetchChild = 0;
        }
        String url = MessageFormat.format(wwUserList, accessToken, String.valueOf(departmentId), fetchChild);

        ResponseEntity<UserListResp> forEntity = restTemplate.getForEntity(url, UserListResp.class);

        UserListResp result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result.getUserList();
    }

    /**
     * 修改用户所在部门
     *
     * @param accessToken
     * @param userId
     * @param department
     * @throws WeworkException
     */
    public void updateDepartment(@NonNull String accessToken, @NonNull String userId, @NonNull List<Integer> department) throws WeworkException {
        String url = MessageFormat.format(wwUserUpdate, accessToken);

        Map<String, Object> map = new HashMap<>();
        map.put("userid", userId);
        map.put("department", department);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(map, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 修改用户所在部门
     *
     * @param accessToken
     * @param userInfo
     * @throws WeworkException
     */
    public void update(@NonNull String accessToken, @NonNull UserInfo userInfo) throws WeworkException {
        String url = MessageFormat.format(wwUserUpdate, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<UserInfo> httpEntity = new HttpEntity<>(userInfo, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * 创建成员
     *
     * @param accessToken
     * @param userInfo
     * @throws WeworkException
     */
    public void create(@NonNull String accessToken, @NonNull UserInfo userInfo) throws WeworkException {
        String url = MessageFormat.format(wwUserCreate, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<UserInfo> httpEntity = new HttpEntity<>(userInfo, headers);
        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

}
