/**
 * kuaikeguanjia.com Inc.
 * Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.entity.provider.LoginInfo;
import cn.kinyun.wework.sdk.entity.provider.ProviderToken;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.google.common.collect.Maps;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;

/**
 * @title ProviderApi
 * @author yanmaoyuan
 * @date 2021年4月30日
 * @version 1.0
 */
@Component
public class ProviderApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;
    
    @Value("${qyapi.service.get_provider_token}")
    private String wwGetProviderToken;

    @Value("${qyapi.service.get_login_info}")
    private String wwGetLoginInfo;

    /**
     * 获取服务商凭证
     * 
     * @param corpId 服务商的corpid
     * @param providerSecret 服务商的secret，在服务商管理后台可见
     * @return
     * @see https://open.work.weixin.qq.com/api/doc/90001/90143/91200
     */
    @GenIgnore
    public ProviderToken getProviderToken(@NonNull String corpId, @NonNull String providerSecret) throws WeworkException {
        
        String url = wwGetProviderToken;
        
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("corpid", corpId);
        params.put("provider_secret", providerSecret);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ProviderToken> response = restTemplate.postForEntity(url, request, ProviderToken.class);

        ProviderToken result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取登录用户信息
     * 
     * @param providerAccessToken 授权登录服务商的网站时，使用应用提供商的provider_access_token，获取方法参见服务商的凭证
     * @param authCode oauth2.0授权企业微信管理员登录产生的code，最长为512字节。只能使用一次，5分钟未被使用自动过期
     * @return
     * @see https://open.work.weixin.qq.com/api/doc/90001/90143/91125
     * @see https://open.work.weixin.qq.com/api/doc/90001/90143/91200
     */
    @GenIgnore
    public LoginInfo getLoginInfo(@NonNull String providerAccessToken, @NonNull String authCode) throws WeworkException {
        String url = MessageFormat.format(wwGetLoginInfo, providerAccessToken);
        
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("auth_code", authCode);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<LoginInfo> response = restTemplate.postForEntity(url, request, LoginInfo.class);

        LoginInfo result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }
}