package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenIgnore;
import cn.kinyun.wework.sdk.entity.Ticket;
import cn.kinyun.wework.sdk.entity.jssdk.JsSignReq;
import cn.kinyun.wework.sdk.entity.jssdk.JsSignResp;
import cn.kinyun.wework.sdk.exception.WeworkException;
import cn.kinyun.wework.sdk.crypt.AesException;
import cn.kinyun.wework.sdk.crypt.SHA1;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信JSSDK接口
 * 
 * @link https://open.work.weixin.qq.com/api/doc/90000/90136/90506
 * @title JsSdkApi
 * @desc 微信JSSDK接口
 * @author yanmaoyuan
 * @date 2019年4月29日
 * @version 1.0
 */
@Slf4j
@Component
public class JsSdkApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.get_ticket}")
    private String wwGetTicket;

    @Value("${qyapi.get_jsapi_ticket}")
    private String wwGetJsapiTicket;

    /**
     * 获得jsapi的临时票据
     * 
     * @param accessToken 接口调用凭证
     * @return jsapi临时票据
     * @throws WeworkException
     */
    public Ticket getAgentTicket(@NonNull String accessToken) throws WeworkException {
        return getTicketByType(accessToken, "agent_config");
    }

    /**
     * 获得接口临时票据
     * 
     * @link https://open.work.weixin.qq.com/api/doc/90000/90136/90506#签名算法
     * @param accessToken 接口调用凭证
     * @param type 临时票据类型 (jsapi, agent_config)
     * @return 临时票据
     * @throws WeworkException
     */
    public Ticket getTicketByType(@NonNull String accessToken, @NonNull String type) throws WeworkException {
        log.info("get ticket with type={}", type);
        // 发起请求
        String url = MessageFormat.format(wwGetTicket, accessToken, type);
        ResponseEntity<Ticket> response = restTemplate.getForEntity(url, Ticket.class);

        Ticket ticket = response.getBody();
        WeworkException.isSuccess(ticket);// 处理错误码
        return ticket;
    }
    
    /**
     * 获得接口临时票据
     * 
     * @link https://open.work.weixin.qq.com/api/doc/90000/90136/90506#签名算法
     * @param accessToken 接口调用凭证
     * @return 临时票据
     * @throws WeworkException
     */
    public Ticket getJsapiTicket(@NonNull String accessToken) throws WeworkException {
        log.info("get jsapi ticket");
        // 发起请求
        String url = MessageFormat.format(wwGetJsapiTicket, accessToken);
        ResponseEntity<Ticket> response = restTemplate.getForEntity(url, Ticket.class);
        
        Ticket ticket = response.getBody();
        WeworkException.isSuccess(ticket);// 处理错误码
        return ticket;
    }

    /**
     * 使用jsapi_ticket和appid对url进行签名，用于JSSDK认证。
     * 
     * @link https://open.work.weixin.qq.com/api/doc/90000/90136/90515
     * @link https://open.work.weixin.qq.com/api/doc/90000/90136/90506
     * @param data 需要签名的url、nonceStr、timestamp等数据
     * @param jsApiTicket jsapi临时票据
     * @return 签名结果
     */
    @GenIgnore
    public JsSignResp signature(@NonNull JsSignReq data, @NonNull String jsApiTicket) {
        log.info("signature for data={}", data);

        String url = data.getUrl();
        String nonceStr = data.getNonceStr();
        String timestamp = data.getTimestamp();

        // 步骤1. 将这些参数使用URL键值对的格式 （即 key1=value1&key2=value2…）拼接成字符串string1。
        // 有两个注意点：1. 字段值采用原始值，不要进行URL转义；2. 必须严格按照如下格式拼接，不可变动字段顺序。
        // jsapi_ticket=JSAPITICKET&noncestr=NONCESTR&timestamp=TIMESTAMP&url=URL
        StringBuilder sb = new StringBuilder(64);
        sb.append("jsapi_ticket=").append(jsApiTicket)
            .append("&noncestr=").append(nonceStr)
            .append("&timestamp=").append(timestamp)
            .append("&url=").append(url);

        // 步骤2. 对字符串进行sha1签名，得到signature
        String signature = null;
        try {
            signature = SHA1.sha1(sb.toString());
        } catch (AesException e) {
            e.printStackTrace();
        }

        // 生成返回结果
        JsSignResp sign = new JsSignResp();
        sign.setNonceStr(nonceStr);
        sign.setTimestamp(timestamp);
        sign.setSignature(signature);

        return sign;
    }
}
