/**
 * kuaikeguanjia.com Inc.
 * Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.chat.ChatData;
import cn.kinyun.wework.sdk.entity.chat.MediaData;
import cn.kinyun.wework.sdk.entity.chat.PermitUserDto;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.google.common.collect.Maps;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;

import lombok.NonNull;

/**
 * 会话存档API
 * 
 * <p>这个API是试验性质的，目的是不通过SDK的中转、直接像企业微信服务器发送HTTP请求拉取会话存档。请不要使用。</p>
 * 
 * @title FinanceApi
 * @author yanmaoyuan
 * @date 2021年3月17日
 * @version 1.0
 * @link https://developer.work.weixin.qq.com/document/path/91614
 */
@Component
public class FinanceApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.finance.getchatdata}")
    private String wwGetChatData;

    @Value("${qyapi.finance.getchatmediadata}")
    private String wwGetMediaData;

    @Value("${qyapi.finance.getPermitUserList}")
    private String getPermitUserList;

    /**
     * 获取企业开启会话内容存档的成员列表
     * @param accessToken
     * @param type 拉取对应版本的开启成员列表。1表示办公版；2表示服务版；3表示企业版。非必填，不填写的时候返回全量成员列表
     * @return
     */
    public PermitUserDto getPermitUserList(@NonNull String accessToken, @GenNullable Integer type) throws WeworkException {
        String url = MessageFormat.format(getPermitUserList, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        if (Objects.nonNull(type)) {
            params.put("type", type);
        }

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<PermitUserDto> response = restTemplate.postForEntity(url, request, PermitUserDto.class);

        PermitUserDto result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 获取会话记录数据
     * 
     * <p>一次拉取调用上限1000条会话记录，可以通过分页拉取的方式来依次拉取。调用频率不可超过600次/分钟。</p>
     * 
     * @param accessToken 会话存档的accessToken
     * @param seq 本次请求获取消息记录开始的seq值。首次访问填写0，非首次使用上次企业微信返回的最大seq。允许从任意seq重入拉取。Uint64类型，范围0-pow(2,64)-1
     * @param limit 一次拉取的消息条数，最大值1000条，超过1000条会返回错误。uint32类型
     * @return
     */
    public ChatData getChatData(@NonNull String accessToken, @NonNull Long seq, @NonNull Integer limit) throws WeworkException {
        String url = MessageFormat.format(wwGetChatData, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("seq", seq);
        params.put("limit", limit);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ChatData> response = restTemplate.postForEntity(url, request, ChatData.class);

        ChatData result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

    /**
     * 多媒体数据
     * 
     * @param accessToken
     * @param fileId
     * @param authKey
     * @return
     */
    public MediaData getMediaData(@NonNull String accessToken, @NonNull String fileId, @NonNull String authKey) throws WeworkException {
        String url = MessageFormat.format(wwGetMediaData, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("fileid", fileId);
        params.put("authkey", authKey);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<MediaData> response = restTemplate.postForEntity(url, request, MediaData.class);

        MediaData result = response.getBody();
        WeworkException.isSuccess(result);
        return result;
    }

}
