package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenNullable;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.custmized.*;
import cn.kinyun.wework.sdk.exception.WeworkException;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.*;

/**
 * 代开发自建应用相关API，主要是用来转换corpid、userid的一些接口。
 *
 * @see <a href="https://developer.work.weixin.qq.com/document/path/95327">与第三方应用的兼容</a>
 * @author yanmaoyuan
 * @date 2022/2/9
 */
@Slf4j
@Component
public class CustomizedApi {

    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    @Value("${qyapi.service.get_customized_auth_url}")
    private String getCustomizedAuthUrl;

    @Value("${qyapi.service.corpid_to_opencorpid}")
    private String toOpenCorpId;

    @Value("${qyapi.service.userid_to_openuserid}")
    private String toOpenUserId;

    @Value("${qyapi.externalcontact.get_new_external_userid}")
    private String getNewExternalUserId;

    @Value("${qyapi.externalcontact.groupchat.get_new_external_userid}")
    private String getNewChatExternalUserId;

    @Value("${qyapi.service.finish_openid_migration}")
    private String finishOpenidMigration;

    /**
     * 获取带参授权链接
     *
     * <p>该API用于获取代开发自建应用授权链接，用于生成带参临时二维码。</p>
     *
     * @param providerAccessToken 服务商的凭证
     * @param templateIds 代开发自建应用模版ID列表，数量不能超过9个
     * @param state state值，可以填写a-zA-Z0-9，长度不可超过32个字节。当扫带参二维码授权代开发模版时，获取企业永久授权码接口会返回该state值
     * @return 代开发自建应用授权的带参临时二维码
     */
    public CustomizedAuthUrl getCustomizedAuthUrl(@NonNull String providerAccessToken, @NonNull List<String> templateIds, @GenNullable String state) throws WeworkException {
        log.info("get customized auth url, templateIds={}, state={}", templateIds, state);
        String url = MessageFormat.format(getCustomizedAuthUrl, providerAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("templateid_list", templateIds);
        if (state != null) {
            params.put("state", state);
        }

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<CustomizedAuthUrl> response = restTemplate.postForEntity(url, request, CustomizedAuthUrl.class);

        CustomizedAuthUrl result = response.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * corpid转换
     *
     * <p>将明文corpid转换为第三方应用获取的corpid</p>
     * @param providerAccessToken 服务商的凭证
     * @param corpId 待获取的企业ID
     * @return 第三方应用获取的corpid
     */
    public String getOpenCorpId(@NonNull String providerAccessToken, @NonNull String corpId) throws WeworkException {
        log.info("get open corpId={}", corpId);
        String url = MessageFormat.format(toOpenCorpId, providerAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("corpid", corpId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<OpenCorpId> response = restTemplate.postForEntity(url, request, OpenCorpId.class);

        OpenCorpId result = response.getBody();
        WeworkException.isSuccess(result);

        log.info("get open corpId, {} --> {}", corpId, result.getOpenCorpId());
        return result.getOpenCorpId();
    }

    /**
     * userid的转换
     *
     * <p>将自建应用获取的userid转换为第三方应用获取的userid</p>
     *
     * @param accessToken 代开发自建应用或第三方应用的接口凭证
     * @param userIds 获取到的成员ID
     * @return
     */
    public OpenUserIds getOpenUserIds(@NonNull String accessToken, @NonNull List<String> userIds) throws WeworkException {

        String url = MessageFormat.format(toOpenUserId, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("userid_list", userIds);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<OpenUserIds> response = restTemplate.postForEntity(url, request, OpenUserIds.class);

        OpenUserIds result = response.getBody();
        WeworkException.isSuccess(result);

        return result;

    }

    /**
     * 对获取openUserId的接口进行封装，把返回结果变成map。
     *
     * @param accessToken 代开发自建应用或第三方应用的接口凭证
     * @param userIds 获取到的成员ID
     * @return 成员userId和openUserId的映射表
     */
    public Map<String, String> getOpenUserIdMap(@NonNull String accessToken, @NonNull List<String> userIds) throws WeworkException {
        log.info("get open userId map, userIds={}", userIds);
        OpenUserIds result = getOpenUserIds(accessToken, userIds);

        Map<String, String> idMap = new HashMap<>();
        if (result.getOpenUserIdList() == null || result.getOpenUserIdList().size() == 0) {
            log.warn("open user id list is empty.");
            return idMap;
        }

        for (OpenUserId it : result.getOpenUserIdList()) {
            idMap.put(it.getUserId(), it.getOpenUserId());
        }
        return idMap;
    }

    /**
     * 对获取openUserId的接口进行封装，获取单个userId的openUserId
     *
     * @param accessToken 代开发自建应用或第三方应用的接口凭证
     * @param userId 获取到的成员ID
     * @return 成员的openUserId
     */
    public String getOpenUserId(@NonNull String accessToken, @NonNull String userId) throws WeworkException {
        log.info("get open userId, userId={}", userId);
        OpenUserIds result = getOpenUserIds(accessToken, Collections.singletonList(userId));
        if (result.getOpenUserIdList() == null || result.getOpenUserIdList().size() == 0) {
            log.warn("open user id list is empty");
            return null;
        }

        return result.getOpenUserIdList().get(0).getOpenUserId();
    }

    /**
     * 转换external_userid
     *
     * <p>将升级前的external_userid转换为升级后的external_userid，仅历史授权的应用可调用，升级后的应用不可调用。</p>
     *
     * @param accessToken 代开发自建应用或第三方应用的接口凭证。
     * @param externalUserIds 旧外部联系人id列表，建议200个，最多不超过1000个。
     * @return 转换后的外部联系人ID
     * @link https://developer.work.weixin.qq.com/document/path/95865
     */
    public OpenExternalUserIds getNewExternalUserId(@NonNull String accessToken, @NonNull List<String> externalUserIds) throws WeworkException {
        String url = MessageFormat.format(getNewExternalUserId, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("external_userid_list", externalUserIds);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<OpenExternalUserIds> response = restTemplate.postForEntity(url, request, OpenExternalUserIds.class);

        OpenExternalUserIds result = response.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 转换群成员external_userid
     *
     * <p>将升级前的external_userid转换为升级后的external_userid，仅历史授权的应用可调用，升级后的应用不可调用。</p>
     *
     * @param accessToken 代开发自建应用或第三方应用的接口凭证。
     * @param chatId 客户群ID
     * @param externalUserIds 旧外部联系人id列表，建议200个，最多不超过1000个。
     * @return 转换后的群成员ID
     * @link https://developer.work.weixin.qq.com/document/path/95865
     */
    public OpenExternalUserIds getNewChatExternalUserId(@NonNull String accessToken, @NonNull String chatId, @NonNull List<String> externalUserIds) throws WeworkException {
        String url = MessageFormat.format(getNewChatExternalUserId, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("chat_id", chatId);
        params.put("external_userid_list", externalUserIds);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<OpenExternalUserIds> response = restTemplate.postForEntity(url, request, OpenExternalUserIds.class);

        OpenExternalUserIds result = response.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * 代开发应用设置迁移完成
     *
     * <p>服务商完成了代开发应用的新旧openid（userid/corpid/external_userid)的迁移，即可主动将该企业设置为“迁移完成”，设置之后，该代开发应用获取到的将是新的openid。注意，该接口需要使用provider_access_token来调用。</p>
     *
     * @param providerAccessToken 应用提供商的provider_access_token
     * @param corpId 企业corpid
     * @param agentId 企业应用id
     * @param openidTypes openid类型：1-userid与corpid; 3-external_userid
     * @link https://developer.work.weixin.qq.com/document/path/95865
     */
    public void finishOpenidMigration(@NonNull String providerAccessToken, @NonNull String corpId, @NonNull Integer agentId, @NonNull List<Integer> openidTypes) throws WeworkException {
        String url = MessageFormat.format(finishOpenidMigration, providerAccessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("corpid", corpId);
        params.put("agentid", agentId);
        params.put("openid_type", openidTypes);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        ErrorCode result = response.getBody();
        WeworkException.isSuccess(result);
    }
}
