package cn.kinyun.wework.sdk.api;

import cn.kinyun.wework.sdk.annotation.GenAsync;
import cn.kinyun.wework.sdk.entity.ErrorCode;
import cn.kinyun.wework.sdk.entity.agent.userid.ExternalUserId;
import cn.kinyun.wework.sdk.entity.agent.userid.UserIds;
import cn.kinyun.wework.sdk.exception.WeworkException;
import cn.kinyun.wework.sdk.entity.agent.Agent;
import cn.kinyun.wework.sdk.entity.agent.AgentList;
import cn.kinyun.wework.sdk.entity.agent.AgentMsgBase;
import cn.kinyun.wework.sdk.entity.agent.SendMsgResult;
import cn.kinyun.wework.sdk.entity.agent.SetAgentParams;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import lombok.NonNull;

/**
 * 应用管理
 */
@Component
public class AgentApi {
    @Autowired
    @Qualifier("weworkRestTemplate")
    private RestTemplate restTemplate;

    /**
     * 获取指定的应用详情
     */
    @Value("${qyapi.agent.get}")
    private String wwAgentGet;

    /**
     * 获取access_token对应的应用列表
     */
    @Value("${qyapi.agent.list}")
    private String wwAgentList;

    /**
     * 设置应用
     */
    @Value("${qyapi.agent.set}")
    private String wwAgentSet;

    /**
     * 发送应用消息
     */
    @Value("${qyapi.message.send}")
    private String wwMessageSend;

    /**
     * 密文external_userid转明文
     */
    @Value("${qyapi.service.openuserid_to_userid}")
    private String openUserId2UserId;

    /**
     * 密文external_userid转明文
     */
    @Value("${qyapi.externalcontact.from_service_external_userid}")
    private String openExternalUserId2externalUserId;


    /**
     * @param accessToken
     * @param agentId 应用id
     * @desc 获取指定的应用详情
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/90227
     */
    public Agent getAgent(@NonNull String accessToken, @NonNull Integer agentId) throws WeworkException {
        String url = MessageFormat.format(wwAgentGet, accessToken);
        url = url + "&agentid=" +agentId;

        ResponseEntity<Agent> forEntity = restTemplate.getForEntity(url, Agent.class);

        Agent result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param accessToken
     * @desc 获取access_token对应的应用列表
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/90227
     */
    public AgentList getAgentList(@NonNull String accessToken) throws WeworkException {
        String url = MessageFormat.format(wwAgentList, accessToken);

        ResponseEntity<AgentList> forEntity = restTemplate.getForEntity(url, AgentList.class);

        AgentList result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * @param params
     * @desc 设置应用
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/90228
     */
    public void setAgent(@NonNull String accessToken, @NonNull SetAgentParams params) throws WeworkException {
        String url = MessageFormat.format(wwAgentSet, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<SetAgentParams> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<ErrorCode> forEntity = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = forEntity.getBody();
        WeworkException.isSuccess(result);
    }

    /**
     * @param accessToken
     * @param msg
     * @desc 发送应用消息
     * @link https://open.work.weixin.qq.com/api/doc/90000/90135/90236
     */
    @GenAsync
    public SendMsgResult sendMsg(@NonNull String accessToken, @NonNull AgentMsgBase msg) throws WeworkException {
        String url = MessageFormat.format(wwMessageSend, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AgentMsgBase> httpEntity = new HttpEntity<>(msg, headers);

        ResponseEntity<SendMsgResult> forEntity = restTemplate.postForEntity(url, httpEntity, SendMsgResult.class);

        SendMsgResult result = forEntity.getBody();
        WeworkException.isSuccess(result);

        return result;
    }



    /**
     * userid转换
     *
     * <p>将代开发应用或第三方应用获取的密文open_userid转换为明文userid。</p>
     *
     * @param accessToken 企业自建应用或基础应用的调用接口凭证。获取方法查看“获取access_token”
     * @param openUserIdList open_userid列表，最多不超过1000个。必须是source_agentid对应的应用所获取
     * @param sourceAgentId 企业授权的代开发自建应用或第三方应用的agentid
     */
    public UserIds convertOpenUserId(@NonNull String accessToken, @NonNull List<String> openUserIdList, @NonNull Integer sourceAgentId) throws WeworkException {
        String url = MessageFormat.format(openUserId2UserId, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("open_userid_list", openUserIdList);
        params.put("source_agentid", sourceAgentId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<UserIds> response = restTemplate.postForEntity(url, request, UserIds.class);

        UserIds result = response.getBody();
        WeworkException.isSuccess(result);

        return result;
    }

    /**
     * userid转换
     *
     * <p>将代开发应用或第三方应用获取的密文open_userid转换为明文userid。</p>
     *
     * @param accessToken 企业自建应用或基础应用的调用接口凭证。获取方法查看“获取access_token”
     * @param externalUserId 服务商主体的external_userid，必须是source_agentid对应的应用所获取
     * @param sourceAgentId 企业授权的代开发自建应用或第三方应用的agentid
     */
    public String convertOpenExternalUserId(@NonNull String accessToken, @NonNull String externalUserId, @NonNull Integer sourceAgentId) throws WeworkException {
        String url = MessageFormat.format(openExternalUserId2externalUserId, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        Map<String, Object> params = Maps.newHashMap();
        params.put("external_userid", externalUserId);
        params.put("source_agentid", sourceAgentId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ExternalUserId> response = restTemplate.postForEntity(url, request, ExternalUserId.class);

        ExternalUserId result = response.getBody();
        WeworkException.isSuccess(result);

        return result.getExternalUserId();
    }
}
