package cn.kinyun.pay.business.dto.request;

import cn.kinyun.pay.business.enums.PayChannelCode;
import cn.kinyun.pay.business.enums.PayChannelType;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Data
@SuppressWarnings({"java:S1066", "common-java:DuplicatedBlocks"})
public class PayConfigModReq extends PayConfigBaseReq implements Serializable {

    private String num;

    /**
     * 支付商户名称
     */
    private String mchName;

    /**
     * 支付商户秘钥
     */
    private String secret;

    /**
     * alipayAppPrivateKey, 仅配置支付宝渠道需要
     */
    private String alipayAppPrivateKey;

    private List<PayCertBase64Dto> payCerts;

    private Long operatorId;

    private static final Set<String> WEIXIN_CERT_NAMES = Sets.newHashSet("apiclient_cert.p12", "apiclient_cert.pem", "apiclient_key.pem");

    private static final Set<String> ALIPAY_CERT_NAMES = Sets.newHashSet("alipayCertPublicKey_RSA2.crt", "alipayRootCert.crt");

    private static final Pattern PATTERN = Pattern.compile("\\d{5,}");

    public void validate(Integer payChannelType, String appId) {
        super.validate();
        Preconditions.checkArgument(StringUtils.isNotBlank(num), "num can not be null or empty");
        Preconditions.checkArgument(StringUtils.isNotBlank(mchName) || StringUtils.isNotBlank(secret) || !CollectionUtils.isEmpty(payCerts) || StringUtils.isNotBlank(alipayAppPrivateKey), "mchName and secret and payCert and alipayAppPrivateKey can not be null or empty at same time");
        Preconditions.checkArgument(Objects.nonNull(operatorId), "operatorId 不能为空");
        if (StringUtils.isNotBlank(secret)) {
            if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_WEIXIN.getValue())) {
                Preconditions.checkArgument(StringUtils.length(secret) == 32, "微信支付秘钥secret为32位数字，请检查");
            }
        }
        if (StringUtils.isNotBlank(alipayAppPrivateKey)) {
            if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
                Preconditions.checkArgument(StringUtils.isNotBlank(alipayAppPrivateKey), "alipayAppPrivateKey不能为空");
            }
        }

        if (CollectionUtils.isNotEmpty(payCerts)) {
            if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_WEIXIN.getValue())) {
                Preconditions.checkArgument(CollectionUtils.size(payCerts) == 3, "微信支付证书有3个文件");
            } else if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
                Preconditions.checkArgument(CollectionUtils.size(payCerts) == 3, "支付宝支付证书有3个文件");
            }
            for (PayCertBase64Dto payCert : payCerts) {
                payCert.validate();
            }
            if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
                // 支付宝证书文件名校验
                Map<String, PayCertBase64Dto> fileName2Dto = payCerts.stream().collect(Collectors.toMap(PayCertBase64Dto::getCertName, Function.identity()));
                StringBuilder stringBuilder = new StringBuilder();
                for (String alipayCertName : ALIPAY_CERT_NAMES) {
                    PayCertBase64Dto dto = fileName2Dto.get(alipayCertName);
                    if (Objects.isNull(dto)) {
                        stringBuilder.append(alipayCertName).append(',');
                    }
                }
                boolean findAppCertPublicKey = false;
                for (PayCertBase64Dto payCert : payCerts) {
                    if (payCert.getCertName().matches("appCertPublicKey_\\d+\\.crt")) {
                        findAppCertPublicKey = true;
                        Matcher matcher = PATTERN.matcher(payCert.getCertName());
                        if (matcher.find()) {
                            Preconditions.checkArgument(StringUtils.equals(matcher.group(), appId), "支付宝应用公钥证书和appId不匹配");
                        }
                    }
                }
                if (!findAppCertPublicKey) {
                    stringBuilder.append("appCertPublicKey_xxxxxx.crt").append(',');
                }
                String errorMsg = "";
                if (StringUtils.isNotBlank(stringBuilder.toString())) {
                    errorMsg = stringBuilder.substring(0, stringBuilder.length() - 1);
                    errorMsg += "证书文件不存在，请检查";
                }
                Preconditions.checkArgument(StringUtils.isEmpty(errorMsg), errorMsg);
            }
            if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_WEIXIN.getValue())) {
                // 微信证书文件名校验
                Set<String> fileNames = payCerts.stream().map(PayCertBase64Dto::getCertName).collect(Collectors.toSet());
                Collection<String> notMatchFileNames = CollectionUtils.subtract(WEIXIN_CERT_NAMES, fileNames);
                if (CollectionUtils.isNotEmpty(notMatchFileNames)) {
                    StringBuilder stringBuilder = new StringBuilder();
                    for (String notMatchFileName : notMatchFileNames) {
                        stringBuilder.append(notMatchFileName).append(',');
                    }
                    String errorMsg = stringBuilder.substring(0, stringBuilder.length() - 1);
                    Preconditions.checkArgument(StringUtils.isEmpty(errorMsg), errorMsg + "证书文件不存在，请检查");
                }
            }
        }
    }

    public String findAlipayAppCert(Integer payChannelType) {
        String appCertName = null;
        if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
            for (PayCertBase64Dto payCert : payCerts) {
                if (payCert.getCertName().matches("appCertPublicKey_\\d+\\.crt")) {
                    appCertName = payCert.getCertName();
                }
            }
        }
        return appCertName;
    }
}
