package cn.kinyun.pay.business.dto.request;

import cn.kinyun.pay.business.enums.PayChannelCode;
import cn.kinyun.pay.business.enums.PayChannelType;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Data
@SuppressWarnings({"common-java:DuplicatedBlocks"})
public class PayConfigAddReq extends PayConfigBaseReq implements Serializable {
    /**
     * scrm等其他系统的商户名称
     */
    private String businessName;
    /**
     * 支付商户号
     */
    private String mchId;
    /**
     * 支付商户名称
     */
    private String mchName;
    /**
     * 支付能力
     */
    private Integer payChannelType;
    /**
     * 微信商户是支付的secret，支付宝商户是authToken
     */
    private String secret;

    /**
     * alipayAppPrivateKey, 仅配置支付宝渠道需要
     */
    private String alipayAppPrivateKey;
    /**
     * 支付证书；微信有三个证书，一般命名为：apiclient_cert.p12,apiclient_cert.pem,apiclient_key.pem, ；支付宝有3个证书,命名分别为：alipayCertPublicKey_RSA2.crt,alipayRootCert.crt,appCertPublicKey_xxxxxxxxxxxxx.crt
     */
    private List<PayCertBase64Dto> payCerts;
    /**
     * 应用appId,如：微信商户的：公众号ID，小程序ID, APP移动应用ID；支付宝商户的应用ID
     */
    private String appId;
    /**
     * 操作人
     */
    private Long operatorId;

    private static final Set<String> WEIXIN_CERT_NAMES = Sets.newHashSet("apiclient_cert.p12", "apiclient_cert.pem", "apiclient_key.pem");

    private static final Set<String> ALIPAY_CERT_NAMES = Sets.newHashSet("alipayCertPublicKey_RSA2.crt", "alipayRootCert.crt");

    private static final Pattern PATTERN = Pattern.compile("\\d{5,}");

    public void validate() {
        super.validate();
        Preconditions.checkArgument(StringUtils.isNotBlank(businessName) && StringUtils.length(businessName) <= 32, "businessName is null or empty, and length no more then 32");
        Preconditions.checkArgument(StringUtils.isNotBlank(mchId) && StringUtils.length(mchId) <= 20, "mchId can not be null or empty,and length no more then 20");
        Preconditions.checkArgument(StringUtils.isNotBlank(mchName) && StringUtils.length(mchName) <= 64, "mchName can not be null or empty, and length no more then 64");
        Preconditions.checkArgument(Objects.nonNull(payChannelType) && Objects.nonNull(PayChannelType.get(payChannelType)), "payChannelType can not be null or empty");
        Preconditions.checkArgument(!CollectionUtils.isEmpty(payCerts), "payCerts can not be null or empty");
        Preconditions.checkArgument(StringUtils.isNotBlank(appId) && StringUtils.length(appId) <= 64, "appId can not be null or empty, and length no more then 64");
        if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_WEIXIN.getValue())) {
            Preconditions.checkArgument(StringUtils.isNotBlank(secret) && StringUtils.length(secret) == 32, "微信支付秘钥secret为32位数字，请检查");
            Preconditions.checkArgument(CollectionUtils.size(payCerts) == 3, "微信支付证书有3个文件");
        } else if (Objects.equals(PayChannelType.get(payChannelType).getServiceCode(), PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
            Preconditions.checkArgument(CollectionUtils.size(payCerts) == 3, "支付宝支付证书有3个文件");
            Preconditions.checkArgument(StringUtils.isNotBlank(alipayAppPrivateKey), "alipayAppPrivateKey不能为空");
        }
        for (PayCertBase64Dto payCert : payCerts) {
            payCert.validate();
        }
        if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
            // 支付宝证书文件名校验
            Map<String, PayCertBase64Dto> fileName2Dto = payCerts.stream().collect(Collectors.toMap(PayCertBase64Dto::getCertName, Function.identity()));
            StringBuilder stringBuilder = new StringBuilder();
            for (String alipayCertName : ALIPAY_CERT_NAMES) {
                PayCertBase64Dto dto = fileName2Dto.get(alipayCertName);
                if (Objects.isNull(dto)) {
                    stringBuilder.append(alipayCertName).append(',');
                }
            }
            boolean findAppCertPublicKey = false;
            for (PayCertBase64Dto payCert : payCerts) {
                if (payCert.getCertName().matches("appCertPublicKey_\\d+\\.crt")) {
                    findAppCertPublicKey = true;
                    Matcher matcher = PATTERN.matcher(payCert.getCertName());
                    if (matcher.find()) {
                        Preconditions.checkArgument(StringUtils.equals(matcher.group(), appId), "支付宝应用公钥证书和appId不匹配");
                    }
                }
            }
            if (!findAppCertPublicKey) {
                stringBuilder.append("appCertPublicKey_xxxxxx.crt").append(',');
            }
            String errorMsg = "";
            if (StringUtils.isNotBlank(stringBuilder.toString())) {
                errorMsg = stringBuilder.substring(0, stringBuilder.length() - 1);
                errorMsg += "证书文件不存在，请检查";
            }
            Preconditions.checkArgument(StringUtils.isEmpty(errorMsg), errorMsg);
        }
        if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_WEIXIN.getValue())) {
            // 微信证书文件名校验
            Set<String> fileNames = payCerts.stream().map(PayCertBase64Dto::getCertName).collect(Collectors.toSet());
            Collection<String> notMatchFileNames = CollectionUtils.subtract(WEIXIN_CERT_NAMES, fileNames);
            if (CollectionUtils.isNotEmpty(notMatchFileNames)) {
                StringBuilder stringBuilder = new StringBuilder();
                for (String notMatchFileName : notMatchFileNames) {
                    stringBuilder.append(notMatchFileName).append(',');
                }
                String errorMsg = stringBuilder.substring(0, stringBuilder.length() - 1);
                Preconditions.checkArgument(StringUtils.isEmpty(errorMsg), errorMsg + "证书文件不存在，请检查");
            }
        }
        Preconditions.checkArgument(Objects.nonNull(operatorId), "operatorId 不能为空");
    }

    public String findAlipayAppCert() {
        String appCertName = null;
        if (PayChannelType.get(payChannelType).getServiceCode().equals(PayChannelCode.PAY_CHANNEL_CODE_ALIPAY.getValue())) {
            for (PayCertBase64Dto payCert : payCerts) {
                if (payCert.getCertName().matches("appCertPublicKey_\\d+\\.crt")) {
                    appCertName = payCert.getCertName();
                }
            }
        }
        return appCertName;
    }
}
