package cn.kinyun.pay.business.component;

import cn.kinyun.pay.business.dto.request.CertValidReq;
import cn.kinyun.pay.business.dto.response.BaseResponse;
import cn.kinyun.pay.business.dto.response.CertResponse;
import cn.kinyun.pay.business.service.PayInnerCertService;
import cn.kinyun.pay.business.utils.DigitalSignatureUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

@Slf4j
@Component
@SuppressWarnings("java:S2696")
public class PayCertBackgroundService {

    @Autowired(required = false)
    private PayInnerCertService certService;

    private static String privateKey;

    private static final ScheduledExecutorService EXECUTOR_SERVICE = Executors.newSingleThreadScheduledExecutor();

    @PostConstruct
    public void init() {
        EXECUTOR_SERVICE.scheduleWithFixedDelay(() -> {

            if (Objects.isNull(certService)) {
                log.info("certService is null,return");
                return;
            }

            if (StringUtils.isNotBlank(privateKey)) {
                // 读取证书,调用接口判断是否发生变化
                CertValidReq req = new CertValidReq();
                try {
                    req.setPublicKey(DigitalSignatureUtil.calPublicKeyFromPrivateKey(privateKey));
                    BaseResponse<Boolean> response = certService.certValid(req);
                    if (response.isSuccess() && response.getData()) {
                        log.info("check cert success");
                        return;
                    }
                } catch (Exception e) {
                    log.error("cal cert fail,", e);
                }

            }
            // 不存在直接调用服务拉取
            reloadPrivateKey();

        }, 20, 5 * 60l, TimeUnit.SECONDS);
        log.info("payCertBackgroundService start");
    }

    /**
     * 证书验签失败时,请主动调用该方法
     */
    public void reloadPrivateKey() {
        BaseResponse<CertResponse> certResponse = certService.queryCurrentCert();
        if (certResponse.isSuccess()) {
            privateKey = certResponse.getData().getPrivateKey();
        }
    }

    /**
     * 对参数计算签名
     *
     * @param req
     * @return
     */
    public String getSign(Object req) {
        while (StringUtils.isEmpty(privateKey)) {
            reloadPrivateKey();
            try {
                Thread.sleep(500L);
            } catch (InterruptedException e) {
                log.error("error: ", e);
                Thread.currentThread().interrupt();
            }
        }
        return DigitalSignatureUtil.signFromSha1WithRSA(req, privateKey);
    }

}
