package cn.kinyun.pay.business.dto.request;


import cn.kinyun.pay.business.dto.request.trans.BaseAccountInfo;
import cn.kinyun.pay.business.enums.PayTransChannelType;
import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.util.Objects;

@ToString
@Data
public class TransRequest extends BaseRequest {

    /**
     * 转账金额，单位为分
     */
    private String amount;

    /**
     * 关联的业务订单编号
     */
    private String bizOrderNum;

    /**
     * 业务转账单号,(唯一)
     */
    private String bizTransNum;

    /**
     * 转账原因
     */
    private String refundReason;

    /**
     * 转账类型
     * <p>
     * 取值为: {@link cn.kinyun.pay.business.enums.PayTransChannelType}
     */
    private Integer payTransType;

    /**
     * 账号信息
     *
     * 当payTransType=101 或 104 (支付宝渠道)时, 使用BaseAccountInfo 的子类 {@link cn.kinyun.pay.business.dto.request.trans.AliPayAccount}
     * 当payTransType=102(微信)时, 使用BaseAccountInfo 的子类 {@link cn.kinyun.pay.business.dto.request.trans.WXPayAccount}
     * 当payTransType=103(银行)时, 使用BaseAccountInfo 的子类 {@link cn.kinyun.pay.business.dto.request.trans.BankCardAccount}
     */
    private BaseAccountInfo accountInfo;

    @Override
    public void validate() {
        super.validate();
        Preconditions.checkArgument(StringUtils.isNotBlank(bizTransNum), "bizTransNum不能为空");
        Preconditions.checkArgument(StringUtils.isNotBlank(amount) && amount.matches("\\d+"), "无效的amount,金额必须以分为单位,且是一个正整数");
        Long totalAmountNum = NumberUtils.toLong(amount);
        Preconditions.checkArgument(totalAmountNum >= 10 && totalAmountNum < 100000000000L, "amount取值范围为:[10,100000000000)");
        Preconditions.checkArgument(Objects.nonNull(payTransType) && PayTransChannelType.get(payTransType) != null, "转账类型不能为空");
        Preconditions.checkArgument(Objects.nonNull(accountInfo), "转账账户不能为空");
        accountInfo.validate();
        Preconditions.checkArgument(StringUtils.isNotBlank(bizOrderNum), "bizOrderNum 不能为空");
        Preconditions.checkArgument(StringUtils.isNotBlank(refundReason), "refundReason 不能为空");
    }
}
