package cn.kinyun.pay.business.dto.request;

import cn.kinyun.pay.business.dto.vo.OrderHeaderDto;
import cn.kinyun.pay.business.enums.PayChannelType;
import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

/**
 * Created by jiangtiteng on 2018/12/6
 */
@Data
@ToString
public class OrderRequest extends BaseRequest {
    /**
     * 渠道类型
     */
    private Integer channelType;

    /**
     * 总金额以分为单位
     */
    private String totalAmount;
    /**
     * 标题
     */
    private String subject;

    /**
     * 支付过期时间(分), 默认30分钟后支付过期
     */
    private Integer timeoutMinute;

    /**
     * 扩展参数
     */
    private String params;

    /**
     * 业务订单, 拆单使用
     */
    private OrderHeaderDto orderHeaderDto;

    @Override
    public void validate() {
        super.validate();
        Preconditions.checkArgument(channelType != null && PayChannelType.get(channelType) != null, "无效的channelType");
        Preconditions.checkArgument(StringUtils.isNotBlank(totalAmount) && totalAmount.matches("\\d+"), "无效的totalAmount,金额必须以分为单位,且是一个正整数");
        Long totalAmountNum = NumberUtils.toLong(totalAmount);
        Preconditions.checkArgument(totalAmountNum > 0 && totalAmountNum < 100000000000L, "totalAmount取值范围为:(0,100000000000)");
        Preconditions.checkArgument(orderHeaderDto != null, "orderHeaderDto 不能为空");
        Preconditions.checkArgument(StringUtils.isNotBlank(subject), "subject 商品简单描述不能为空");
        Long amountNum = orderHeaderDto.validate();
        Preconditions.checkArgument(amountNum >= totalAmountNum, "本次支付金额totalAmount不能大于订单总金额amount");
        if (timeoutMinute == null) {
            timeoutMinute = 30;
        } else {
            Preconditions.checkArgument(timeoutMinute >= 5 && timeoutMinute <= 120, "timeoutMinute取值范围为[5,120]");
        }
    }
}
