/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package com.kuaike.skynet.logic.wechat.utils;

import com.kuaike.common.entity.WechatMessage;
import com.kuaike.skynet.logic.wechat.msg.BaseAppMsg;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;

import org.apache.commons.lang3.StringUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * 派发微信消息
 * 
 * @title IMsgHandler
 * @author yanmaoyuan
 * @date 2019年12月9日
 * @version 1.0
 */
@Slf4j
public abstract class AbstractMsgHandler {

    /**
     * 用于解析xml文件
     */
    protected XmlMapper xmlMapper;

    public AbstractMsgHandler() {
        xmlMapper = new XmlMapper();
        xmlMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    }

    public void dispatch(WechatMessage wechatMessage) {
        log.info("dispatch msg type={}, talkerType={}, isSend={}, requestId={}", wechatMessage.getType(),
            wechatMessage.getTalkerType(), wechatMessage.getIsSend(), wechatMessage.getRequestId());

        switch (wechatMessage.getType()) {
            // 文本
            case 1: {
                String message = wechatMessage.getMessage();

                // FIXME 这种消息，是否应该单独配置一个回复？
                // 注意，这本身不是一个消息，而是一个微信系统事件，表示有人把当前微信添加成了好友。
                // 前端把这个Event当作消息上报给了后台。
                if (message != null && message.contains("ticket") && message.contains("opcode")
                    && message.contains("scene")) {
                    onWaitJoinFriendEvent(wechatMessage);
                    return;
                }

                text(wechatMessage);
                return;
            }
            // 图片
            case 3:
            case 23: // 没有收到过
            case 33: // 没有收到过
            {
                image(wechatMessage);
                return;
            }
            // 不知道是什么类型，从来没有收到过。
            case 11: // 没有收到过
            case 36: // 没有收到过
            {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 语音
            case 34: {
                voice(wechatMessage);
                return;
            }
            // 摇一摇朋友%s向你打招呼
            // 漂流瓶朋友%s向你打招呼
            case 37: // 没有收到过
            {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 未知类型
            case 13:
            case 39: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 个人名片、公众号名片
            case 42:
            case 66: {
                visitingCard(wechatMessage);
                return;
            }
            // 视频
            case 43:
            case 62: {
                video(wechatMessage);
                return;
            }
            // GIF动画
            case 47:
            case 0x00100031: {
                gif(wechatMessage);
                return;
            }
            // 48: 定位
            case 48: {
                location(wechatMessage);
                return;
            }
            // 应用消息
            case 49:
            case 0x01000031:
            case 0x10000031:
            case 0x21000031:
            case 0x23000031:
            case 0x29000031: {
                xmlAppMsg(wechatMessage);
                return;
            }
            // 小程序
            case 523: 
            case 526: {
                miniProgram(wechatMessage);
                return;
            }
            // 语音通话
            case 50:
            case 53: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 视频通话
            case 52:
            case 12299999: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 未知类型
            case 55:
            case 57: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 通过扫码入群
            case 0x11000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 快递，麦当劳点餐
            case 0x13000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 书/专辑/电影/商品
            case 0x14000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 卡/卡券
            case 0x16000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 电视
            case 0x18000031: {
                // TODO television
                unknown(wechatMessage);
                return;
            }
            // 419430449: 转账信息
            case 0x19000031: {
                transferMoney(wechatMessage);
                return;
            }
            // 436207665, 469762097: 红包
            case 0x1A000031:
            case 0x1C000031: {
                redPack(wechatMessage);
                return;
            }
            // 卡包(滴滴出行发票)
            case 0x1B000031: {
                // TODO card_msg_item_digest
                unknown(wechatMessage);
                return;
            }
            // 未知
            case 0x1E000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            // 卡包
            case 0x1F000031: {
                // TODO ard_msg_item_digest
                unknown(wechatMessage);
                return;
            }
            // 未知
            case 0x20000031: {
                // TODO honey_pay_prefix
                unknown(wechatMessage);
                return;
            }
            // 570425393: 邀请加入群聊
            case 0x22000031: {
                onJoinGroupEvent(wechatMessage);
                return;
            }
            // 未知
            case 0x24000031: {
                // TODO
                unknown(wechatMessage);
                return;
            }
            case 0x28000031: {
                // TODO downloaderapp_download_game_success_tips, downloaderapp_welcome_short
                unknown(wechatMessage);
                return;
            }
            // 10000: 系统消息
            case 10000: {
                sysMsg(wechatMessage);
                return;
            }
            // 10002: XXX想邀请XXX加入群
            case 10002: {
                chatRoomAccessVerifyApplication(wechatMessage);
                return;
            }
            // 49 - 5 收到了应用链接，可能是超链接卡片，也可能是加群链接卡片
            // 这个495是社群客户端在上报数据时，直接把 49 和 5这两个类型拼起来的结果，不是实际类型。
            // 实际类型是49，内部xml的type为5
            case 495: {
                appUrl(wechatMessage);
                return;
            }
            default:
                // 未知的消息类型，打日志。
                unknown(wechatMessage);
                break;
        }
    }

    /**
     * 文本
     * 
     * @param wechatMessaage
     */
    public void text(WechatMessage wechatMessage) {
        log.info("text is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 图片
     * 
     * @param wechatMessaage
     */
    public void image(WechatMessage wechatMessage) {
        log.info("image is not processed, requestId={}, talkerId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getTalkerId());
    }

    /**
     * 语音
     * 
     * @param wechatMessage
     */
    public void voice(WechatMessage wechatMessage) {
        log.info("voice is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 视频
     * 
     * @param wechatMessage
     */
    public void video(WechatMessage wechatMessage) {
        log.info("video is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 个人号名片、公众号名片
     * 
     * @param wechatMessage
     */
    public void visitingCard(WechatMessage wechatMessage) {
        log.info("visiting card is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getTalkerId());
    }

    /**
     * GIF动画
     * 
     * @param wechatMessage
     */
    public void gif(WechatMessage wechatMessage) {
        log.info("gif is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 系统消息
     * 
     * 系统消息有很多不同的内容，需要根据内容来处理。
     * 
     * @param wechatMessage
     */
    protected void sysMsg(WechatMessage wechatMessage) {
        String message = wechatMessage.getMessage();

        if (MsgUtils.match(message, MsgUtils.FRIEND_IS_JOINED_PATTERN)) {// 判断是否为添加好友消息
            //目前测试的收到这个系统消息的都是被动加好友
            onJoinedByFriendEvent(wechatMessage);
        } else if (MsgUtils.match(message, MsgUtils.JOINED_BY_FRIEND_PATTERN)) {// 判断是否为被动添加好友消息
            onJoinedByFriendEvent(wechatMessage);
        } else if (MsgUtils.match(message, MsgUtils.GROUP_YOU_ARE_OWNER_PATTERN)) {// 成为了群主
            onBecomeOwnerEvent(wechatMessage);
        } else if (MsgUtils.match(message, MsgUtils.GROUP_YOU_ARE_KICKED_PATTERN)) {// 被踢出群
            onKickedEvent(wechatMessage);
        } else {
            onUnknownSysMsg(wechatMessage);
        }
    }

    /**
     * 收到某群成员邀请另一个人入群的申请，需要群主验证。
     * 
     * @param wechatMessage
     */
    public void chatRoomAccessVerifyApplication(WechatMessage wechatMessage) {
        log.info("Recv ChatRoomAccessVerifyApplication is not processed, requestId={}, wechatId={}, message={}", wechatMessage.getRequestId(),
            wechatMessage.getWechatId(), wechatMessage.getMessage());
    }

    /**
     * 转账消息
     * 
     * 转账消息只会发生在好友之间，不会发生在群聊中。
     * 
     * @param wechatMessage
     */
    public void transferMoney(WechatMessage wechatMessage) {
        log.info("transfer money is not processed, requestId={}", wechatMessage.getRequestId());
        // TODO 这里需要解析参数
        int type = 0;// TODO 暂时使用常量0表示
        // Object obj = null;
        switch (type) {
            case 1:
                // 内部还有判断
                // if (obj == null) {
                // appmsg_remittance_digest_receiver_wait
                // } else {
                // appmsg_remittance_digest_payer_wait
                // }
                break;
            case 3:
                // if (obj == null) {
                // appmsg_remittance_digest_payer_collected
                // } else {
                // appmsg_remittance_digest_payer_collected
                // }
                break;
            case 4:
                // if (obj == null) {
                // appmsg_remittance_digest_payer_refused
                // } else {
                // appmsg_remittance_digest_receiver_refused
                // }
                break;
            default: {
                break;
            }
        }
    }

    /**
     * 红包
     * 
     * 红包消息在群聊、私聊都会存在。
     * 
     * @param wechatMessage
     */
    public void redPack(WechatMessage wechatMessage) {
        log.info("red pack is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 定位
     * 
     * @param wechatMessage
     */
    public void location(WechatMessage wechatMessage) {
        log.info("location is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 接收摇一摇/漂流瓶消息
     * 
     * @param wechatMessage
     */
    protected void xmlShakeMsg(WechatMessage wechatMessage) {
        log.info("dispatch shake hand msg, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
        // TODO 需要解析xml才能获取具体的信息
        int sence = 0;// TODO 暂时使用常量0表示
        switch (sence) {
            case 18:
                // fmt_say_hello_to_you
                break;
            case 22:
            case 23:
            case 24:
            case 26:
            case 27:
            case 28:
            case 29:
                // fmt_shake_say_hello_to_you
                break;
            case 25:
                // fmt_bottle_say_hello_to_you
                break;
            default:
                // fmt_want_to_be_your_friend
                break;
        }
        unknown(wechatMessage);
    }

    /**
     * 接收带有子类型的消息
     * 
     * @param wechatMessage
     */
    protected void xmlAppMsg(WechatMessage wechatMessage) {
        log.info("dispatch appmsg, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());

        if (StringUtils.isBlank(wechatMessage.getMessage())) {
            log.warn("XmlAppMsg is blank, requestId={}, isSend={}", wechatMessage.getRequestId(), wechatMessage.getIsSend());
            return;
        }

        // 需要解析xml才能得到具体的消息类型
        int type = 0;
        BaseAppMsg msg = null;
        try {
            msg = xmlMapper.readValue(wechatMessage.getMessage(), BaseAppMsg.class);
            type = msg.getAppmsg().getType();
            log.info("appmsg={}", msg);
        } catch (Exception e) {
            log.error("解析应用消息失败, requestId={}, xml={}", wechatMessage.getRequestId(), wechatMessage.getMessage(), e);
        }

        switch (type) {
            // 文本
            case 1:
                // getTitle
                onUnknownAppMsg(wechatMessage);
                return;
            // 图片
            case 2:
                // appPic
                onUnknownAppMsg(wechatMessage);
                return;
            // 音乐
            case 3:
                // appMusic
                onUnknownAppMsg(wechatMessage);
                return;
            // 视频
            case 4:
                // appVideo
                onUnknownAppMsg(wechatMessage);
                return;
            // 邀请入群的url
            case 5:
                // appUrl
                appUrl(wechatMessage);
                return;
            // 文件
            case 6:
                // appFile
                file(wechatMessage);
                return;
            // 应用消息
            case 7:
                // appApp
                onUnknownAppMsg(wechatMessage);
                return;
            // 表情
            case 8:
                // appEmoji
                onUnknownAppMsg(wechatMessage);
                return;
            // 表情分享
            case 15:
            case 26:
            case 27:
                // appEmojiShare
                onUnknownAppMsg(wechatMessage);
                return;
            // 卡包
            case 16:
                // cardMsgItemDigest
                onUnknownAppMsg(wechatMessage);
                return;
            case 17:
                // 未知类型
                onUnknownAppMsg(wechatMessage);
                return;
            // 聊天记录
            case 19:
                // appRecord
                onUnknownAppMsg(wechatMessage);
                return;
            // 笔记
            case 24:
                // appNote
                onUnknownAppMsg(wechatMessage);
                return;
            // 艺术家推荐
            case 25:
                // appDesignerShare
                onUnknownAppMsg(wechatMessage);
                return;
            // 小程序
            case 33:
                miniProgram(wechatMessage);
                return;
            case 36:
                // app_brand_public abstract_name_with_brackets
                miniProgram(wechatMessage);
                return;
            // wxcard 卡包
            case 34:
                // card_msg_item_digest
                onUnknownAppMsg(wechatMessage);
                return;
            // 应用消息
            case 40:
                onUnknownAppMsg(wechatMessage);
                return;
            // 应用消息
            case 44:
                onUnknownAppMsg(wechatMessage);
                return;
            // 应用消息
            case 46:
                onUnknownAppMsg(wechatMessage);
                return;
            default:
                onUnknownAppMsg(wechatMessage);
                break;
        }
    }

    /**
     * 应用链接，可能包含加群卡片
     * 
     * @param wechatMessage
     * @return
     */
    public void appUrl(WechatMessage wechatMessage) {
        log.info("AppUrl is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }
    
    /**
     * 文件
     * 
     * @param wechatMessage
     * @return
     */
    public void file(WechatMessage wechatMessage) {
        log.info("File is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 微信小程序
     * 
     * @param wechatMessage
     */
    public void miniProgram(WechatMessage wechatMessage) {
        log.info("MiniProgram is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(), wechatMessage.getTalkerId());
    }

    /**
     * 加入群聊消息
     * 
     * @param wechatMessage
     */
    public void onJoinGroupEvent(WechatMessage wechatMessage) {
        log.info("JoinGroupEvent is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getTalkerId());
    }

    /**
     * 有微信用户申请添加我为好友，等待同意。
     * 
     * @param wechatMessage
     */
    public void onWaitJoinFriendEvent(WechatMessage wechatMessage) {
        log.info("WaitJoinFriendEvent is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 主动添加好友完成事件
     * 
     * @param wechatMessage
     */
    public void onFriendIsJoinedEvent(WechatMessage wechatMessage) {
        log.info("FriendIsJoinedEvent is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 被微信用户添加为好友了
     * 
     * @param wechatMessage
     */
    public void onJoinedByFriendEvent(WechatMessage wechatMessage) {
        log.info("JoinedByFriendEvent is not processed, requestId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getTalkerId());
    }

    /**
     * 成为了群主
     * 
     * @param wechatMessage
     */
    public void onBecomeOwnerEvent(WechatMessage wechatMessage) {
        log.info("BecomeOwnerEvent is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 被从群里踢出来了
     * 
     * @param wechatMessage
     */
    public void onKickedEvent(WechatMessage wechatMessage) {
        log.info("KickedEvent is not processed, requestId={}", wechatMessage.getRequestId());
    }

    /**
     * 未知的消息
     * 
     * @param wechatMessage
     */
    public void unknown(WechatMessage wechatMessage) {
        log.info("Receive unknown msg, requestId={}, type={}, wechatId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getType(), wechatMessage.getWechatId(), wechatMessage.getTalkerId());
    }

    /**
     * 未知的应用消息
     * 
     * @param wechatMessage
     * @param appMsg
     */
    public void onUnknownAppMsg(WechatMessage wechatMessage) {
        log.info("Receive unknown appmsg, requestId={}, type={}, wechatId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getType(), wechatMessage.getWechatId(), wechatMessage.getTalkerId());
    }

    /**
     * 未知的系统消息
     * 
     * @param wechatMessage
     */
    public void onUnknownSysMsg(WechatMessage wechatMessage) {
        log.info("Receive unknown sysmsg, requestId={}, type={}, wechatId={}, talkerId={}", wechatMessage.getRequestId(),
            wechatMessage.getType(), wechatMessage.getWechatId(), wechatMessage.getTalkerId());
    }
}
