/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.exception;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;

import org.apache.commons.lang3.StringUtils;

import java.util.ResourceBundle;

import lombok.NonNull;

/**
 * @title WeixinException
 * @desc 微信接口异常
 * @author yanmaoyuan
 * @date 2019年4月28日
 * @version 1.0
 */
public class WeixinException extends RuntimeException {

    private static final long serialVersionUID = 3738216579645492071L;

    // 异常消息格式
    private static final String MESSAGE_FORMAT = "error(%d):%s";

    // 当errorCode没有找到对应的错误信息时，使用此默认信息。
    private static final String DEFAULT_ERROR_MESSAGE = "请求微信接口失败，未知错误。";

    // 国际化存储的微信错误码
    private static ResourceBundle RESOURCE_BUNDLE = ResourceBundle.getBundle("message.WeixinErrorCode");

    /**
     * 根据错误码，获得错误信息。
     *
     * @param errorCode 错误码
     * @return 错误信息
     */
    public static String getErrorMsg(Integer errorCode) {
        try {
            return RESOURCE_BUNDLE.getString(String.valueOf(errorCode));
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 统一处理错误码，判断请求是否成功，如果请求不成功则抛出异常。
     *
     * @param errorCode 错误码
     * @return 请求成功返回true，失败抛出异常。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public static boolean isSuccess(@NonNull Integer errorCode) throws WeixinException {
        return isSuccess(errorCode, DEFAULT_ERROR_MESSAGE);
    }

    /**
     * 统一处理错误码，判断请求是否成功，如果请求不成功则抛出异常。
     *
     * @param error 微信返回的错误码
     * @return 请求成功返回true，失败抛出异常。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public static boolean isSuccess(@NonNull ErrorCode error) throws WeixinException {
        return isSuccess(error.getErrCode(), error.getErrMsg());
    }

    /**
     * 统一处理错误码，判断请求是否成功，如果请求不成功则抛出异常。
     *
     * @param errorCode 错误码
     * @param defaultErrorMsg 默认错误信息。如果是无法识别的错误码，将会使用默认错误提示信息。
     * @return 请求成功返回true，失败抛出异常。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public static boolean isSuccess(Integer errorCode, String defaultErrorMsg)
            throws WeixinException {
        // 正常消息
        if (errorCode == null || errorCode == 0) {
            return true;
        }

        // 从资源文件中查找错误信息
        String errorMsg = getErrorMsg(errorCode);

        // 如果资源文件中没有定义这个错误码，则使用默认的错误提示信息。
        if (errorMsg == null) {
            if (StringUtils.isNoneBlank(defaultErrorMsg)) {
                errorMsg = defaultErrorMsg;
            } else {
                errorMsg = DEFAULT_ERROR_MESSAGE;
            }
        }

        // 抛出异常
        throw new WeixinException(errorCode, errorMsg);
    }

    /**
     * 错误码
     */
    private Integer errorCode;

    /**
     * 错误信息
     */
    private String errorMsg;

    public WeixinException(Integer errorCode, String errorMsg) {
        super(String.format(MESSAGE_FORMAT, errorCode, errorMsg));
        this.errorCode = errorCode;
        this.errorMsg = errorMsg;
    }

    public Integer getErrorCode() {
        return errorCode;
    }

    public String getErrorMsg() {
        return errorMsg;
    }
}