/**
 * kuaike.com Inc.
 * Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.entity.message.resp;

import cn.kinyun.scrm.weixin.sdk.entity.message.BaseMsg;
import cn.kinyun.scrm.weixin.sdk.utils.MsgXmlBuilder;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * 回复用户消息
 * 
 * 当用户发送消息给公众号时（或某些特定的用户操作引发的事件推送时），会产生一个POST请求，开发者可以在响应包（Get）中返回特定XML结构，来对该消息进行响应（现支持回复文本、图片、图文、语音、视频、音乐）。严格来说，发送被动响应消息其实并不是一种接口，而是对微信服务器发过来消息的一次回复。
 * 
 * @title BaseRespMsg
 * @desc 回复用户消息
 * @author yanmaoyuan
 * @date 2019年4月25日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140543">被动回复用户消息</a>
 */
@Data
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
public abstract class BaseRespMsg extends BaseMsg {

    private static final long serialVersionUID = 3550839891058945212L;

    /**
     * 是否允许群发转载文章
     * 
     * <p>
     * 群发接口新增 send_ignore_reprint 参数，开发者可以对群发接口的 send_ignore_reprint 参数进行设置，指定待群发的文章被判定为转载时，是否继续群发。
     * </p>
     * <p>
     * 当 send_ignore_reprint 参数设置为1时，文章被判定为转载时，且原创文允许转载时，将继续进行群发操作。
     * </p>
     * <p>
     * 当 send_ignore_reprint 参数设置为0时，文章被判定为转载时，将停止群发操作。
     * </p>
     * <p>
     * send_ignore_reprint 默认为0。
     * </p>
     */
    private Integer sendIgnoreReprint;

    /**
     * 开发者侧群发msgid，长度限制64字节，如不填，则后台默认以群发范围和群发内容的摘要值做为clientmsgid。
     * 
     * <p>
     * 开发者调用群发接口时可以主动设置 clientmsgid 参数，避免重复推送。
     * </p>
     * <p>
     * 群发时，微信后台将对 24 小时内的群发记录进行检查，如果该 clientmsgid 已经存在一条群发记录，则会拒绝本次群发请求，返回已存在的群发msgid，开发者可以调用“查询群发消息发送状态”接口查看该条群发的状态。
     * </p>
     * 
     */
    private String clientMsgId;

    /**
     * 将消息内容转化为XML格式
     * 
     * @return XML数据文本
     */
    public String toXml() {
        MsgXmlBuilder builder = new MsgXmlBuilder();

        builder.addCData("ToUserName", toUserName);
        builder.addCData("FromUserName", fromUserName);
        builder.addTag("CreateTime", String.valueOf(System.currentTimeMillis() / 1000));
        builder.addCData("MsgType", msgType);

        return builder.toString();
    }
}