package cn.kinyun.scrm.weixin.sdk.api.shop;


import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.RegisterShopReq;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.SubmitBasicInfoReq;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.SubmitMerchantInfoReq;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.CheckAuditStatusResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;

/**
 * 微信小商店开店api
 *
 * @author zhangjing
 * @version 1.0
 * @title WxShopRegisterAPI
 * @desc 微信小商店开店api
 * @date 2021年12月30日
 * @see <a href="https://developers.weixin.qq.com/miniprogram/dev/platform-capabilities/business-capabilities/ministore/minishopopencomponent/API/register/introduction.html">微信小商店开店api</a>
 */
@Slf4j
@Component
public class WxShopRegisterAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 异步状态查询 POST
     */
    @Value("${wx.shop.register.check_audit_status}")
    private String checkAuditStatusPost;

    /**
     * 注册小商店账号 POST
     */
    @Value("${wx.shop.register.register_shop}")
    private String registerShopPost;

    /**
     * 提交支付资质 POST
     */
    @Value("${wx.shop.register.submit_merchantinfo}")
    private String submitMerchantInfoPost;

    /**
     * 提交小商店基础信息 POST
     */
    @Value("${wx.shop.register.submit_basicinfo}")
    private String submitBasicInfoPost;

    /**
     * 异步状态查询
     *
     * @param accessToken 接口调用凭证
     * @param wxName      微信号
     * @return
     */
    public CheckAuditStatusResp checkAuditStatus(@NonNull String accessToken, @NonNull String wxName) {
        log.info("checkAuditStatus with wxName={}", wxName);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(wxName), "wxName不能为空");

        ResponseEntity<CheckAuditStatusResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("wx_name", wxName);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(checkAuditStatusPost, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CheckAuditStatusResp.class);

        CheckAuditStatusResp result = resp.getBody();
        log.info("checkAuditStatus result={}",result);
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 注册小商店账号
     *
     * @param componentAccessToken 接口调用凭证
     * @param req                  注册参数
     * @return
     */
    public ErrorCode registerShop(@NonNull String componentAccessToken, @NonNull RegisterShopReq req) {
        log.info("registerShop with req={}", req);

        Preconditions.checkArgument(StringUtils.isNotBlank(componentAccessToken), "componentAccessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<RegisterShopReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(registerShopPost, componentAccessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        log.info("registerShop result={}",result);
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 提交支付资质
     *
     * @param accessToken 接口调用凭证
     * @param req         资质参数
     * @return
     */
    public ErrorCode submitMerchantInfo(@NonNull String accessToken, @NonNull SubmitMerchantInfoReq req) {
        log.info("submitMerchantInfo with req={}", req);

        Preconditions.checkArgument(StringUtils.isNotBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SubmitMerchantInfoReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(submitMerchantInfoPost, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        log.info("submitMerchantInfo result={}",result);
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 提交小商店基础信息
     *
     * @param accessToken 接口调用凭证
     * @param req         参数
     * @return
     */
    public ErrorCode submitBasicinfo(@NonNull String accessToken, @NonNull SubmitBasicInfoReq req) {
        log.info("submitBasicinfo with req={}", req);

        Preconditions.checkArgument(StringUtils.isNotBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SubmitBasicInfoReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(submitBasicInfoPost, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        log.info("submitBasicinfo result={}",result);
        WeixinException.isSuccess(result);
        return result;
    }
}
