package cn.kinyun.scrm.weixin.sdk.entity.menu.dto;

import cn.kinyun.scrm.weixin.sdk.enums.menu.MenuButtonType;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.kuaike.common.errorcode.CommonErrorCode;
import com.kuaike.common.exception.BusinessException;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.List;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.ToString;

@NoArgsConstructor
@Data
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
public class MenuButton extends MenuButtonBase {
    /**
     * 必填字段 菜单响应动作类型 {@link MenuButtonType}
     */
    @JsonProperty("type")
    String type;
    /**
     * click等点击类型必填
     *
     * 菜单KEY值，用于消息接口推送，不超过128字节
     */
    @JsonProperty("key")
    String key;

    /**
     * view、miniprogram类型必填
     *
     * 网页 链接，用户点击菜单可打开链接，不超过1024字节。 type为miniprogram时，不支持小程序的老版本客户端将打开本url。
     */
    @JsonProperty("url")
    String url;
    /**
     * miniprogram类型必填
     *
     * 小程序的appid（仅认证公众号可配置）
     */
    @JsonProperty("appid")
    String appId;
    /**
     * media_id类型和view_limited类型必须
     *
     * 调用新增永久素材接口返回的合法media_id
     */
    @JsonProperty("media_id")
    String mediaId;

    /**
     * 开放平台素材库的id
     */
    private Long matId;

    /**
     * miniprogram类型必须
     *
     * 小程序的页面路径
     */
    @JsonProperty("pagepath")
    String pagePath;
    /**
     * 二级菜单数组，个数应为1~5个
     */
    @JsonProperty("sub_button")
    List<MenuButton> subButton;

    /**
     * 素材类型,自用
     */
    String mediaType;

    public void validate() {
        Preconditions.checkArgument(StringUtils.isNotEmpty(name), "name is null");
        // 如果子菜单为空,则需要校验type
        if (CollectionUtils.isEmpty(subButton)) {
            checkType();
        } else {
            subButton.forEach(a -> a.validate());
        }
    }

    private void checkType() {
        Preconditions.checkArgument(StringUtils.isNotEmpty(type), "type is null");
        try {
            MenuButtonType.valueOf(type);
        } catch (Exception e) {
            throw new BusinessException(CommonErrorCode.PARAM_ERROR, "type is not a valid value");
        }
        if (MenuButtonType.view.name().equals(type)) {
            Preconditions.checkArgument(StringUtils.isNotEmpty(url), "菜单名称为" + name + "的网页地址url为空");
        } else if (MenuButtonType.click.name().equals(type)) {
            Preconditions.checkArgument(StringUtils.isNotEmpty(key), "key is null");
        } else if (MenuButtonType.media_id.name().equals(type)) {
            Preconditions.checkArgument(matId != null || StringUtils.isNotEmpty(mediaId), "素材id为空");
        }
    }

    /**
     * 基础构造方法
     * 
     * @param type
     * @param name
     * @param key
     * @param url
     * @param appid
     * @param media_id
     * @param pagepath
     * @param sub_button
     */
    private MenuButton(String type, String name, String key, String url, String appid, String media_id, String pagepath,
        List<MenuButton> sub_button) {
        super(name);
        this.type = type;
        this.key = key;
        this.url = url;
        this.appId = appid;
        this.mediaId = media_id;
        this.pagePath = pagepath;
        this.subButton = sub_button;
    }

    /**
     * click button
     * 
     * @param name
     * @param key
     */
    public static MenuButton buildClickButton(@NonNull String name, @NonNull String key) {
        return new MenuButton(MenuButtonType.click.name(), name, key, null, null, null, null, null);
    }

    /**
     * view button
     * 
     * @param name
     * @param url
     */
    public static MenuButton buildViewButton(@NonNull String name, @NonNull String url) {
        return new MenuButton(MenuButtonType.view.name(), name, null, url, null, null, null, null);
    }

    /**
     * 小程序 button
     * 
     * @param name
     * @param url
     * @param appid
     * @param pagePath
     * @return
     */
    public static MenuButton buildMiniProgramButton(@NonNull String name, @NonNull String url, @NonNull String appid,
        @NonNull String pagePath) {
        return new MenuButton(MenuButtonType.miniprogram.name(), name, null, url, appid, null, pagePath, null);
    }

    /**
     * 扫码带提示 button
     * 
     * @param name
     * @param key
     * @return
     */
    public static MenuButton buildScancodeWaitMsgButton(@NonNull String name, @NonNull String key,
        List<MenuButton> subButton) {
        return new MenuButton(MenuButtonType.scancode_waitmsg.name(), name, key, null, null, null, null, subButton);
    }

    /**
     * 扫码推事件 button
     * 
     * @param name
     * @param key
     * @return
     */
    public static MenuButton buildScancodePushButton(@NonNull String name, @NonNull String key,
        List<MenuButton> subButton) {
        return new MenuButton(MenuButtonType.scancode_push.name(), name, key, null, null, null, null, subButton);
    }

    /**
     * 系统拍照发图 button
     * 
     * @param name
     * @param key
     * @param subButton
     * @return
     */
    public static MenuButton buildPicSysphotoButton(@NonNull String name, @NonNull String key,
        List<MenuButton> subButton) {
        return new MenuButton(MenuButtonType.pic_sysphoto.name(), name, key, null, null, null, null, subButton);
    }

    /**
     * 拍照或者相册发图 button
     * 
     * @param name
     * @param key
     * @param subButton
     * @return
     */
    public static MenuButton buildPicPhotoOrAlbumButton(@NonNull String name, @NonNull String key,
        List<MenuButton> subButton) {
        return new MenuButton(MenuButtonType.pic_photo_or_album.name(), name, key, null, null, null, null, subButton);
    }

    /**
     * 微信相册发图 button
     * 
     * @param name
     * @param key
     * @param subButton
     * @return
     */
    public static MenuButton buildPicWeiXinButton(@NonNull String name, @NonNull String key,
        List<MenuButton> subButton) {
        return new MenuButton(MenuButtonType.pic_weixin.name(), name, key, null, null, null, null, subButton);
    }

    /**
     * 发送位置 button
     * 
     * @param name
     * @param key
     * @return
     */
    public static MenuButton buildLocationSelectButton(@NonNull String name, @NonNull String key) {
        return new MenuButton(MenuButtonType.location_select.name(), name, key, null, null, null, null, null);
    }

    /**
     * 图片
     * 
     * @param name
     * @param mediaId
     * @return
     */
    public static MenuButton buildMediaId(@NonNull String name, @NonNull String mediaId) {
        return new MenuButton(MenuButtonType.media_id.name(), name, null, null, null, mediaId, null, null);
    }

    /**
     * 图文消息
     * 
     * @param name
     * @param mediaId
     * @return
     */
    public static MenuButton buildViewLimited(@NonNull String name, @NonNull String mediaId) {
        return new MenuButton(MenuButtonType.view_limited.name(), name, null, null, null, mediaId, null, null);
    }

    /**
     * 一级按钮
     * 
     * @param name
     * @param subButton
     * @return
     */
    public static MenuButton buildTopLevelButton(@NonNull String name, List<MenuButton> subButton) {
        return new MenuButton(null, name, null, null, null, null, null, subButton);
    }

    public String toJsonStr() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        String json = null;
        try {
            json = objectMapper.writeValueAsString(this);
        } catch (JsonProcessingException e) {

        }
        return json;
    }

    public static void main(String[] args) {
        MenuButton button = MenuButton.buildClickButton("aaaa", "wqer");
        System.out.println(button.toJsonStr());

        MenuButton button1 = MenuButton.buildScancodePushButton("dsfs", "fdsfs", Lists.newArrayList());
        System.out.println(button1.toJsonStr());

        String json =
            "{\"type\":\"scancodePush\",\"mediaId\":\"mediaaaa\",\"name\":\"dsfs\",\"key\":\"fdsfs\",\"aaa\":[]}";
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        MenuButton button2 = null;
        try {
            button2 = objectMapper.readValue(json, MenuButton.class);
        } catch (IOException e) {

        }
        System.out.println(button2);
    }
}
