/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.custom;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.enums.WxMsgType;
import cn.kinyun.scrm.weixin.sdk.enums.custom.WxTypingStatus;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.kuaike.common.utils.JacksonUtil;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.BaseRespMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.ImageMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.MenuMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.MiniProgramMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.MpNewsMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.MusicMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.NewsMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.TextMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.VideoMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.VoiceMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.WxCardMsg;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信客服消息
 * 
 * @title WxCustomMsgAPI
 * @desc 微信客服消息
 * @author yanmaoyuan
 * @date 2019年4月29日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140547">客服消息</a>
 */
@Slf4j
@Component
public class WxCustomMsgAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 客服发送消息 POST
     */
    @Value("${wx.message.custom.send}")
    private String wxMessageCustomSend;

    /**
     * 客服输入状态 POST
     */
    @Value("${wx.message.custom.typing}")
    private String wxMessageCustomTyping;

    /**
     * 发送客服消息
     * 
     * @param accessToken 接口调用凭证
     * @param message 消息内容
     * @param kfaccount 客服账号，可选。可以指定为某个客服帐号来发消息，在微信6.0.2及以上版本中显示客服的自定义头像。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void sendCustomMsg(@NonNull String accessToken, @NonNull BaseRespMsg message, String kfaccount)
        throws WeixinException {
        log.info("send custom msg with message={}, kfaccount={}", message, kfaccount);

        Preconditions.checkArgument(StringUtils.isNoneBlank(message.getToUserName()), "接受者的openid为空");

        // 校验客服消息类型
        checkCustomMsgType(message);

        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        Map<String, Object> params = Maps.newHashMap();

        // 设定消息的接收者
        params.put("touser", message.getToUserName());
        // 根据消息类型
        params.put("msgtype", message.getMsgType());
        // 填充消息内容
        params.put(message.getMsgType(), message);

        // 如果需要以某个客服帐号来发消息
        if (StringUtils.isNoneBlank(kfaccount)) {
            Map<String, Object> customService = Maps.newHashMap();
            customService.put("kf_account", kfaccount);
            params.put("customservice", customService);
        }

        // 使用 clientmsgid 参数，避免重复推送
        if (message.getClientMsgId() != null) {
            params.put("clientmsgid", message.getClientMsgId());
        }

        // 构造请求
        // 这里直接使用json序列化后getBytes()
        // Jackson在序列化字符串时，先把字符串转换成char[]再逐一序列化。
        // 一个emoji占两个char，会被转化成\\uXXXX\\uXXXX这种形式，在公众号就看不到emoji了。
        byte[] data = JacksonUtil.obj2Str(params).getBytes(StandardCharsets.UTF_8);
        HttpEntity<?> request = new HttpEntity<>(data, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageCustomSend, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        // 处理错误码
        WeixinException.isSuccess(response.getBody());
    }

    /**
     * 设置客服输入状态，此接口需要客服消息接口权限。
     * 
     * <p>
     * 如果不满足发送客服消息的触发条件，则无法下发输入状态。
     * </p>
     * <p>
     * 下发输入状态，需要客服之前30秒内跟用户有过消息交互。
     * </p>
     * <p>
     * 在输入状态中（持续15s），不可重复下发输入态。
     * </p>
     * <p>
     * 在输入状态中，如果向用户下发消息，会同时取消输入状态。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @param openId 用户的openid
     * @param command 输入状态
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void sendTypingStatus(@NonNull String accessToken, @NonNull String openId, @NonNull WxTypingStatus command)
        throws WeixinException {
        log.info("send typing status to openid={}, command={}", openId, command);

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("touser", openId);
        params.put("command", command.name());// 取枚举的名字
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageCustomTyping, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 校验客服消息类型
     * 
     * @param message 消息
     */
    private void checkCustomMsgType(@NonNull BaseRespMsg message) {

        // 校验字段是否为空
        Preconditions.checkArgument(StringUtils.isNoneBlank(message.getMsgType()), "消息类型为空");

        // 校验是否为合法的消息类型
        WxMsgType msgType = WxMsgType.get(message.getMsgType());
        Preconditions.checkArgument(msgType != null, "未知的消息类型");

        // 校验是否允许通过客服发送
        switch (msgType) {
            case Text: {
                Preconditions.checkArgument(message instanceof TextMsg, "消息类型与实例类型不匹配");
                break;
            }
            case Image:
                Preconditions.checkArgument(message instanceof ImageMsg, "消息类型与实例类型不匹配");
                break;
            case Voice:
                Preconditions.checkArgument(message instanceof VoiceMsg, "消息类型与实例类型不匹配");
                break;
            case Video:
                Preconditions.checkArgument(message instanceof VideoMsg, "消息类型与实例类型不匹配");
                break;
            case Music:
                Preconditions.checkArgument(message instanceof MusicMsg, "消息类型与实例类型不匹配");
                break;
            case News:
                Preconditions.checkArgument(message instanceof NewsMsg, "消息类型与实例类型不匹配");
                break;
            case MpNews:
                Preconditions.checkArgument(message instanceof MpNewsMsg, "消息类型与实例类型不匹配");
                break;
            case MsgMenu: {
                Preconditions.checkArgument(message instanceof MenuMsg, "消息类型与实例类型不匹配");
                break;
            }
            case WxCard:
                Preconditions.checkArgument(message instanceof WxCardMsg, "消息类型与实例类型不匹配");
                break;
            case MiniProgramPage:
                Preconditions.checkArgument(message instanceof MiniProgramMsg, "消息类型与实例类型不匹配");
                break;
            default:
                throw new IllegalArgumentException(
                    "客服消息仅支持下列类型：文本(text), 图片(image), 语音(voice), 视频(video), 音乐(music), 图文(news,mpnews), 菜单(msgmenu), 卡券(wxcard), 小程序卡片(miniprogrampage)");
        }
    }
}