package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.user.req.BatchTaggingReq;
import cn.kinyun.scrm.weixin.sdk.entity.user.req.BatchUserInfoReq;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.UserInfo;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.UserInfoListResp;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.UserListResp;
import cn.kinyun.scrm.weixin.sdk.entity.user.resp.UserTagListResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.collect.Maps;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author : shanyu
 * @version V1.0
 * @Project: weixin-platform
 * @Package com.kuaike.weixin.api
 * @Description: 用户管理
 * @date Date : 2019年04月25日 15:15
 */
@Slf4j
@Component
public class WxUserAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.user.getList}")
    private String wxUserGetList;

    @Value("${wx.user.getInfo}")
    private String wxUserGetInfo;

    @Value("${wx.user.batchGetInfo}")
    private String wxUserBatchGetInfo;

    @Value("${wx.user.tag.get}")
    private String wxUserTagGet;

    @Value("${wx.user.tag.batchtagging}")
    private String wxUserTagBatchtagging;

    @Value("${wx.user.tag.batchuntagging}")
    private String wxUserTagBatchuntagging;

    @Value("${wx.user.tag.getIdList}")
    private String wxUserTagGetIdList;

    @Value("${wx.tag.member.getblacklist}")
    private String wxTagMemberGetblacklist;

    @Value("${wx.tag.member.batchblacklist")
    private String wxTagMemberBatchblacklist;

    @Value("${wx.tag.member.batchunblacklist}")
    private String wxTagMemberBatchunblacklist;

    /**
     * 获取粉丝列表（每次拉取上限10000）
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param nextOpenId 下一个粉丝openId （传null表示从头开始拉取）
     * @return 粉丝列表
     * @throws WeixinException
     */
    public UserListResp getList(@NonNull String accessToken, String nextOpenId) throws WeixinException {
        log.info("get user list with next_openid={}", nextOpenId);
        String url = MessageFormat.format(wxUserGetList, accessToken);
        if (nextOpenId != null) {
            url = url + "&next_openid=" + nextOpenId;
        }
        ResponseEntity<UserListResp> response = restTemplate.postForEntity(url, null, UserListResp.class);
        UserListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取用户基本信息
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param openId 普通用户的标识，对当前公众号唯一
     * @param lang 国家地区语言版本
     * @return 用户基本信息
     * @throws WeixinException
     */
    public UserInfo getUserInfo(@NonNull String accessToken, @NonNull String openId, @NonNull String lang)
        throws WeixinException {
        log.info("get user info with openid={}, lang={}", openId, lang);
        String url = MessageFormat.format(wxUserGetInfo, accessToken, openId, lang);
        ResponseEntity<UserInfo> response = restTemplate.postForEntity(url, null, UserInfo.class);

        UserInfo result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量获取用户基本信息
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param batchUserInfoReq 批量获取用户信息的参数
     * @return 用户基本信息
     * @throws WeixinException
     */
    public UserInfoListResp batchGetUserInfo(@NonNull String accessToken, @NonNull BatchUserInfoReq batchUserInfoReq)
        throws WeixinException {
        log.info("batch get user info with userList.size={}",
                CollectionUtils.isNotEmpty(batchUserInfoReq.getUserList())
                    ? batchUserInfoReq.getUserList().size() : 0);
        String url = MessageFormat.format(wxUserBatchGetInfo, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BatchUserInfoReq> request = new HttpEntity<BatchUserInfoReq>(batchUserInfoReq, headers);
        ResponseEntity<UserInfoListResp> response = restTemplate.postForEntity(url, request, UserInfoListResp.class);
        UserInfoListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取对应标签下的粉丝列表
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param tagId 标签id
     * @param nextOpenId //第一个拉取的OPENID，不填默认从头开始拉取
     * @return 粉丝列表
     * @throws WeixinException
     */
    public UserListResp tagGet(@NonNull String accessToken, @NonNull Long tagId, String nextOpenId)
        throws WeixinException {
        log.info("get user list with tag_id={}, next_openid={}", tagId, nextOpenId);
        String url = MessageFormat.format(wxUserTagGet, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("tagid", tagId);
        params.put("next_openid", nextOpenId);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<UserListResp> response = restTemplate.postForEntity(url, request, UserListResp.class);
        UserListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量为粉丝打标签
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param batchTaggingReq 批量打标签参数
     * @throws WeixinException
     */
    public void batchTagging(@NonNull String accessToken, @NonNull BatchTaggingReq batchTaggingReq)
        throws WeixinException {
        log.info("batch tagging with params={}", batchTaggingReq);
        String url = MessageFormat.format(wxUserTagBatchtagging, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BatchTaggingReq> request = new HttpEntity<BatchTaggingReq>(batchTaggingReq, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 批量为粉丝取消标签
     *
     * @param accessToken 授权方accessToken（第三方平台调用）||公众号accessToken（公众号调用）
     * @param batchTaggingReq 批量取消标签参数
     * @throws WeixinException
     */
    public void batchUnTagging(@NonNull String accessToken, @NonNull BatchTaggingReq batchTaggingReq)
        throws WeixinException {
        log.info("batch untagging with params={}", batchTaggingReq);
        String url = MessageFormat.format(wxUserTagBatchuntagging, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BatchTaggingReq> request = new HttpEntity<BatchTaggingReq>(batchTaggingReq, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 获取用户标签列表
     *
     * @param accessToken 接口调用凭证
     * @param openId 粉丝openid
     * @return 用户标签列表
     * @throws WeixinException
     */
    public UserTagListResp userTagGetIdList(@NonNull String accessToken, @NonNull String openId)
        throws WeixinException {
        log.info("get user tag list with openid={}", openId);
        String url = MessageFormat.format(wxUserTagGetIdList, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("openid", openId);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<UserTagListResp> response = restTemplate.postForEntity(url, request, UserTagListResp.class);
        UserTagListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 设置用户备注名
     *
     * @param accessToken 接口调用凭证
     * @param openId 粉丝的openid
     * @param remark 备注名
     * @throws WeixinException
     */
    public void userTagGetIdList(@NonNull String accessToken, @NonNull String openId, @NonNull String remark)
        throws WeixinException {
        log.info("set user remark with openid={}, remark={}", openId, remark);
        String url = MessageFormat.format(wxUserTagGetIdList, accessToken);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("openid", openId);
        params.put("remark", remark);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 获取黑名单列表
     *
     * @param accessToken 接口调用凭证
     * @param beginOpenId
     * @return 黑名单列表
     * @throws WeixinException
     */
    public UserListResp getBlackList(@NonNull String accessToken, String beginOpenId) throws WeixinException {
        log.info("get black list with begin_openid={}", beginOpenId);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        String url = MessageFormat.format(wxTagMemberGetblacklist, accessToken);
        Map<String, Object> params = Maps.newHashMap();
        params.put("begin_openid", beginOpenId);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<UserListResp> response = restTemplate.postForEntity(url, request, UserListResp.class);
        UserListResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量拉黑用户
     *
     * @param accessToken 接口调用凭证
     * @param openIdList 粉丝的openid列表
     * @throws WeixinException
     */
    public void batchBlackList(@NonNull String accessToken, List<String> openIdList) throws WeixinException {
        log.info("batch black list with openid_list={}", openIdList);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        String url = MessageFormat.format(wxTagMemberBatchblacklist, accessToken);
        Map<String, Object> params = Maps.newHashMap();
        params.put("openid_list", openIdList);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 取消拉黑用户
     *
     * @param accessToken 接口调用凭证
     * @param openIdList 粉丝的openid列表
     * @throws WeixinException
     */
    public void batchUnBlackList(@NonNull String accessToken, List<String> openIdList) throws WeixinException {
        log.info("batch unblack list with openid_list={}", openIdList);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        String url = MessageFormat.format(wxTagMemberBatchunblacklist, accessToken);
        Map<String, Object> params = Maps.newHashMap();
        params.put("openid_list", openIdList);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }

}
