package cn.kinyun.scrm.weixin.sdk.api.menu;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.menu.req.CreateCustomerMenuReq;
import cn.kinyun.scrm.weixin.sdk.entity.menu.req.CustomizedMenuTestReq;
import cn.kinyun.scrm.weixin.sdk.entity.menu.req.DeleteCustomerMenuReq;
import cn.kinyun.scrm.weixin.sdk.entity.menu.req.MenuCreateEntity;
import cn.kinyun.scrm.weixin.sdk.entity.menu.resp.AllMenuResp;
import cn.kinyun.scrm.weixin.sdk.entity.menu.resp.CreateCustomizedResp;
import cn.kinyun.scrm.weixin.sdk.entity.menu.resp.CustomizedMenuConfResp;
import cn.kinyun.scrm.weixin.sdk.entity.menu.resp.CustomizedMenuTestResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.kuaike.common.utils.JacksonUtil;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @ClassName MenuAPI
 * @Description
 * @Author zhangjing
 * @DATE 2019/4/25 16:41
 * @Version 1.0
 */
@Slf4j
@Component
public class MenuAPI {

    @Autowired
    RestTemplate restTemplate;

    @Value("${wx.menu.create}")
    String createMenuUrl;

    @Value("${wx.menu.get}")
    String getAllMenuUrl;

    @Value("${wx.menu.delAll}")
    String delAllMenuUrl;

    @Value("${wx.menu.addCustomMenu}")
    String createCustomizedMenuUrl;

    @Value("${wx.menu.delCustomMenu}")
    String delCustomizedMenuUrl;

    @Value("${wx.menu.testMatchResult}")
    String testMatchResultUrl;

    @Value("${wx.menu.getCurrentCustomizedMenuInfo}")
    String currentCustomizedMenuInfoUrl;

    /**
     * 创建菜单
     * @param accessToken
     * @param entity
     * @return
     */
    public ErrorCode createMenu(@NonNull String accessToken, @NonNull MenuCreateEntity entity) {
        log.info("create menu with params={}", entity);
        //validate

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 这里直接使用json序列化后getBytes()
        // 因为RestTemplate中使用的JsonGenerator莫名其妙把emoji转成了unicode
        byte[] data = JacksonUtil.obj2Str(entity).getBytes();
        HttpEntity<?> request = new HttpEntity<>(data, headers);

        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(createMenuUrl, request, ErrorCode.class, accessToken);

        // 处理错误码
        WeixinException.isSuccess(response.getBody());
        return response.getBody();
    }

    /**
     * 查询当前全部的菜单
     * @param accessToken
     * @return
     */
    public AllMenuResp getAllMenu(@NonNull String accessToken){
        log.info("get all menu.");
        ResponseEntity<AllMenuResp> response = restTemplate.getForEntity(getAllMenuUrl, AllMenuResp.class, accessToken);
        WeixinException.isSuccess(response.getBody());
        return response.getBody();
    }

    /**
     * 删除全部菜单
     * @param accessToken
     * @return
     */
    public ErrorCode deleteAllMenu(@NonNull String accessToken){
        log.info("delete all menu.");
        ResponseEntity<ErrorCode> response = restTemplate.getForEntity(delAllMenuUrl,ErrorCode.class,accessToken);
        WeixinException.isSuccess(response.getBody());
        return response.getBody();
    }

    /**
     * 创建个性化菜单
     * @param req
     * @return
     */
    public CreateCustomizedResp createCustomizedMenu(@NonNull String accessToken, @NonNull CreateCustomerMenuReq req){
        log.info("crete customize menu with params={}", req);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);


        // 这里直接使用json序列化后getBytes()
        // 因为RestTemplate中使用的JsonGenerator莫名其妙把emoji转成了unicode
        byte[] data = JacksonUtil.obj2Str(req).getBytes();
        HttpEntity<?> request = new HttpEntity<>(data, headers);

        ResponseEntity<CreateCustomizedResp> response = restTemplate.postForEntity(createCustomizedMenuUrl, request, CreateCustomizedResp.class, accessToken);
        WeixinException.isSuccess(response.getBody());
        return response.getBody();
    }

    /**
     * 删除个性化菜单
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode delCustomizedMenu(@NonNull String accessToken,@NonNull DeleteCustomerMenuReq req){
        log.info("delete customeized menu with params={}", req);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<DeleteCustomerMenuReq> request = new HttpEntity<DeleteCustomerMenuReq>(req, headers);
        ResponseEntity<ErrorCode> responseEntity=restTemplate.postForEntity(delCustomizedMenuUrl, request, ErrorCode.class,accessToken);
        WeixinException.isSuccess(responseEntity.getBody());
        return responseEntity.getBody();
    }

    /**
     * 测试个性化菜单匹配结果
     * @param accessToken
     * @param req
     * @return
     */
    public CustomizedMenuTestResp testMatchResult(@NonNull String accessToken, @NonNull CustomizedMenuTestReq req){
        log.info("test match result with params={}", req);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<CustomizedMenuTestReq> request = new HttpEntity<CustomizedMenuTestReq>(req, headers);
        ResponseEntity<CustomizedMenuTestResp> responseEntity=restTemplate.postForEntity(testMatchResultUrl, request, CustomizedMenuTestResp.class,accessToken);
        WeixinException.isSuccess(responseEntity.getBody());
        return responseEntity.getBody();
    }

    /**
     * 获取自定义菜单配置接口
     * @param accessToken
     */
    public CustomizedMenuConfResp queryCustomizedMenuConf(@NonNull String accessToken){
        log.info("query customized menu conf.");
        ResponseEntity<CustomizedMenuConfResp> responseEntity=restTemplate.getForEntity(currentCustomizedMenuInfoUrl,CustomizedMenuConfResp.class,accessToken);
        WeixinException.isSuccess(responseEntity.getBody());
        return responseEntity.getBody();
    }
}
