package cn.kinyun.scrm.weixin.sdk.api.applet;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.CommitCodeReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.SubmitAuditReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.GetCodePrivacyInfoResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.LatestAuditStatusResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.SubmitAuditResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.UploadMediaResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.kuaike.common.errorcode.CommonErrorCode;
import com.kuaike.common.exception.BusinessException;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.ObjectInputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author zhangjing
 * @desc
 * @date 2022/1/18 10:33 上午
 */
@Slf4j
@Component
public class AppletCodeManageAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 小程序代码管理-上传代码  POST
     */
    @Value("${wx.applet.code_management.commit_code}")
    private String commitCodeUrl;

    /**
     * 小程序代码管理-获取体验版二维码  GET
     */
    @Value("${wx.applet.code_management.getQrcode}")
    private String getQrCodeUrl;

    /**
     * 小程序代码管理-提交审核  POST
     */
    @Value("${wx.applet.code_management.submit_audit}")
    private String submitAuditUrl;

    /**
     * 小程序代码管理-提审素材上传接口  POST/FORM
     */
    @Value("${wx.applet.code_management.upload_media}")
    private String uploadMediaUrl;

    /**
     * 小程序代码管理-查询最新一次提交的审核状态  GET
     */
    @Value("${wx.applet.code_management.get_latest_audit_status}")
    private String getLatestAuditStatusUrl;

    /**
     * 小程序代码管理-发布已通过审核的小程序  POST
     */
    @Value("${wx.applet.code_management.release}")
    private String releaseUrl;

    @Value("${wx.applet.code_management.get_code_privacy_info}")
    private String getCodePrivacyInfoUrl;

    /**
     * 获取代码草稿列表
     *
     * @param accessToken
     * @return
     */
    public ErrorCode commitCode(@NonNull String accessToken, @NonNull CommitCodeReq req) {
        log.info("commitCode with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CommitCodeReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(commitCodeUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    private static final Set<String> images = Sets.newHashSet("png", "jpg", "jpeg", "gif");

    private static final Set<String> videos = Sets.newHashSet("mp4");

    public UploadMediaResp uploadMedia(@NonNull String accessToken, String fileName, File file) {

        log.info("uploadMedia with fileName={}", fileName);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(StringUtils.isNotBlank(fileName), "fileName不能为空");
        Preconditions.checkArgument(Objects.nonNull(file) && file.exists(), "file不能为空");

        String extension = FilenameUtils.getExtension(fileName);
        if (images.contains(extension)) {
            Preconditions.checkArgument(file.length() <= FileUtils.ONE_MB * 2, "图片文件最大2M");
        } else if (videos.contains(extension)) {
            Preconditions.checkArgument(file.length() <= FileUtils.ONE_MB * 10, "视频文件最大10M");
        } else {
            Preconditions.checkArgument(images.contains(extension) || videos.contains(extension), "图片支持PNG,JPEG,JPG,GIF格式,视频支持MP4格式");
        }

        ResponseEntity<UploadMediaResp> resp = null;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", fileName);
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param, headers);
        String url = MessageFormat.format(uploadMediaUrl, accessToken);
        resp = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                UploadMediaResp.class);
        UploadMediaResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 发布已通过审核的小程序
     *
     * @param accessToken
     * @return
     */
    public LatestAuditStatusResp getLatestAuditStatus(@NonNull String accessToken) {
        log.info("getLatestAuditStatus");

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<LatestAuditStatusResp> resp = null;

        // 发送请求
        String url = MessageFormat.format(commitCodeUrl, accessToken);
        resp = restTemplate.getForEntity(url, LatestAuditStatusResp.class);

        LatestAuditStatusResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 发布已通过审核的小程序
     *
     * @param accessToken
     * @return
     */
    public ErrorCode release(@NonNull String accessToken) {
        log.info("release");

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<Map<String, Objects>> httpEntity = new HttpEntity<>(Maps.newHashMap(), headers);

        // 发送请求
        String url = MessageFormat.format(releaseUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }


    /**
     * 获取体验版二维码
     *
     * @param accessToken
     * @param path
     * @return
     */
    public byte[] getQrCode(@NonNull String accessToken, @NonNull String path) {
        log.info("getQrCode with path={}", path);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        if (StringUtils.isNotBlank(path)) {
            try {
                path = URLEncoder.encode(path, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                log.error("path encode failed: {}", path, e);
                throw new BusinessException(CommonErrorCode.PARAM_ERROR, "path编码异常");
            }
        }
        String url = MessageFormat.format(getQrCodeUrl, accessToken, path);
        ResponseEntity<byte[]> resp = restTemplate.getForEntity(url, byte[].class);
        byte[] body = resp.getBody();
        if (body == null) {
            throw new BusinessException(CommonErrorCode.BUSINESS_ERROR, "微信返回结果为空");
        }
        try {
            ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(body);
            ObjectInputStream objectInputStream = new ObjectInputStream(byteArrayInputStream);
            ErrorCode errorCode = (ErrorCode) objectInputStream.readObject();
            WeixinException.isSuccess(errorCode);
        } catch (Exception e) {
            log.error("获取体验版二维码 byte[] 转 ErrorCode 判断是否成功异常", e);
            throw new BusinessException(CommonErrorCode.BUSINESS_ERROR);
        }
        return body;
    }

    /**
     * 提交审核
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SubmitAuditResp submitAudit(@NonNull String accessToken, @NonNull SubmitAuditReq req) {
        log.info("submitAudit with req={}", req);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<SubmitAuditResp> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<SubmitAuditReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(submitAuditUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SubmitAuditResp.class);

        SubmitAuditResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取隐私接口检测结果
     * @param accessToken
     * @return
     */
    public GetCodePrivacyInfoResp getCodePrivacyInfo(@NonNull String accessToken) {
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<GetCodePrivacyInfoResp> resp = null;

        // 发送请求
        String url = MessageFormat.format(getCodePrivacyInfoUrl, accessToken);
        resp = restTemplate.getForEntity(url, GetCodePrivacyInfoResp.class);

        GetCodePrivacyInfoResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
