package cn.kinyun.scrm.weixin.sdk.api.applet;


import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.category.AddCategoryReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.category.DeleteCategoryReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.category.GetAllCategoriesResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.category.GetCategoryResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

/**
 * @author Created by chenjiaqiang on 2022/01/18
 **/
@Slf4j
@Component
public class AppletCategoryAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.applet.category.getAllCategories}")
    private String getAllCategoriesUrl;

    @Value("${wx.applet.category.getCategory}")
    private String getCategoryUrl;

    @Value("${wx.applet.category.addCategory}")
    private String addCategoryUrl;

    @Value("${wx.applet.category.deleteCategory}")
    private String deleteCategoryUrl;


    /**
     * 获取可以设置的所有类目
     *
     * @param accessToken
     * @return
     */
    public GetAllCategoriesResp getAllCategories(@NonNull String accessToken) {
        log.info("getAllCategories with accessToken={}", accessToken);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        // 发送请求
        String url = MessageFormat.format(getAllCategoriesUrl, accessToken);
        ResponseEntity<GetAllCategoriesResp> resp = restTemplate.getForEntity(url, GetAllCategoriesResp.class);
        GetAllCategoriesResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }


    /**
     * 获取已设置的所有类目
     *
     * @param accessToken
     * @return
     */
    public GetCategoryResp getCategory(@NonNull String accessToken) {
        log.info("getCategory with accessToken={}", accessToken);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        // 发送请求
        String url = MessageFormat.format(getCategoryUrl, accessToken);
        ResponseEntity<GetCategoryResp> resp = restTemplate.getForEntity(url, GetCategoryResp.class);
        GetCategoryResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }


    /**
     * 添加类目
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode addCategory(@NonNull String accessToken, @NonNull AddCategoryReq req) {
        log.info("addCategory with req={}", req);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<AddCategoryReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(addCategoryUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 删除类目
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode deleteCategory(@NonNull String accessToken, @NonNull DeleteCategoryReq req) {
        log.info("deleteCategory with req={}", req);
        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        ResponseEntity<ErrorCode> resp = null;
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<DeleteCategoryReq> httpEntity = new HttpEntity<>(req, headers);
        // 发送请求
        String url = MessageFormat.format(deleteCategoryUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;

    }
}
