/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.media.MediaResult;
import cn.kinyun.scrm.weixin.sdk.entity.message.mass.req.DelMassMsg;
import cn.kinyun.scrm.weixin.sdk.entity.message.mass.resp.MsgId;
import cn.kinyun.scrm.weixin.sdk.entity.message.mass.resp.MsgStatus;
import cn.kinyun.scrm.weixin.sdk.entity.message.mass.resp.SendSpeed;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.*;
import cn.kinyun.scrm.weixin.sdk.enums.WxMsgStatus;
import cn.kinyun.scrm.weixin.sdk.enums.WxMsgType;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.kuaike.common.utils.JacksonUtil;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 微信群发消息API，用于给用户发送消息
 * 
 * @title WxMassMsgAPI
 * @desc 微信消息API
 * @author yanmaoyuan
 * @date 2019年4月24日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1481187827_i0l21">群发接口和原创校验</a>
 */
@Slf4j
@Component
@SuppressWarnings("java:S125")
public class WxMassMsgAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private WxMediaAPI wxMediaAPI;

    /**
     * 根据标签进行群发 POST
     */
    @Value("${wx.message.mass.sendall}")
    private String wxMessageMassSendAll;

    /**
     * 根据OpenID列表群发 POST
     */
    @Value("${wx.message.mass.send}")
    private String wxMessageMassSend;

    /**
     * 删除群发消息 POST
     */
    @Value("${wx.message.mass.delete}")
    private String wxMessageMassDelete;

    /**
     * 预览接口 POST
     */
    @Value("${wx.message.mass.preview}")
    private String wxMessageMassPreview;

    /**
     * 查询群发消息发送状态 POST
     */
    @Value("${wx.message.mass.get}")
    private String wxMessageMassGet;

    /**
     * 获取群发速度 POST
     */
    @Value("${wx.message.mass.speed.get}")
    private String wxMessageMassGetSpeed;

    /**
     * 设置群发速度 POST
     */
    @Value("${wx.message.mass.speed.set}")
    private String wxMessageMassSetSpeed;

    /**
     * 群发给所有订阅用户
     * 
     * @param accessToken 接口调用凭证
     * @param message 消息本体
     * @return 群发结果。请注意：在返回成功时，意味着群发任务提交成功，并不意味着此时群发已经结束，所以，仍有可能在后续的发送过程中出现异常情况导致用户未收到消息，如消息有时会进行审核、服务器不稳定等。此外，群发任务一般需要较长的时间才能全部发送完毕，请耐心等待。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public MsgId sendToAll(@NonNull String accessToken, @NonNull BaseRespMsg message) throws WeixinException {
        // 群发意味着tagId为空
        return sendByTag(accessToken, message, "");
    }

    /**
     * 根据标签进行群发【订阅号与服务号认证后均可用】
     * 
     * @param accessToken 接口调用凭证
     * @param message 消息本体
     * @param tagId 群发到的标签的tag_id，参见用户管理中用户分组接口，可根据tag_id发送给指定群组的用户，若不填写tag_id则群发给所有用户
     * @return 群发结果。请注意：在返回成功时，意味着群发任务提交成功，并不意味着此时群发已经结束，所以，仍有可能在后续的发送过程中出现异常情况导致用户未收到消息，如消息有时会进行审核、服务器不稳定等。此外，群发任务一般需要较长的时间才能全部发送完毕，请耐心等待。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public MsgId sendByTag(@NonNull String accessToken, @NonNull BaseRespMsg message, String tagId)
        throws WeixinException {
        log.info("mass send all, message={}, tagId={}", message, tagId);

        // 请求参数
        Map<String, Object> params = Maps.newHashMap();

        // 设定消息的接收者
        Map<String, Object> filter = Maps.newHashMap();

        boolean isToAll = StringUtils.isBlank(tagId);
        filter.put("is_to_all", isToAll);
        if (!isToAll) {
            filter.put("tag_id", tagId);
        }
        params.put("filter", filter);

        // 设置群发消息体
        HttpEntity<?> request = getMassSendRequest(accessToken, params, message);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassSendAll, accessToken);
        ResponseEntity<MsgId> response = restTemplate.postForEntity(url, request, MsgId.class);

        MsgId msgId = response.getBody();
        WeixinException.isSuccess(msgId);// 处理错误码

        return msgId;
    }

    /**
     * 根据OpenID列表群发【订阅号不可用，服务号认证后可用】
     * 
     * @param accessToken 接口调用凭证
     * @param message 消息本体
     * @param openIds 执行微信用户的openId集合。
     * @return 群发结果。请注意：在返回成功时，意味着群发任务提交成功，并不意味着此时群发已经结束，所以，仍有可能在后续的发送过程中出现异常情况导致用户未收到消息，如消息有时会进行审核、服务器不稳定等。此外，群发任务一般需要较长的时间才能全部发送完毕，请耐心等待。
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public MsgId sendByOpenId(@NonNull String accessToken, @NonNull BaseRespMsg message,
        @NonNull Collection<String> openIds) throws WeixinException {
        log.info("mass send by openids, message={}, openIds={}", message, openIds);

        Preconditions.checkArgument(CollectionUtils.isNotEmpty(openIds), "微信接收者的openid不可为空");

        // 对openId去重，并且排除 null 元素。
        Set<String> ids = openIds.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(ids), "微信接收者的openid不可为空");

        // 设定消息的接收者
        Map<String, Object> params = Maps.newHashMap();
        params.put("touser", ids.toArray(new String[0]));

        // 设置群发消息体
        HttpEntity<?> request = getMassSendRequest(accessToken, params, message);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassSend, accessToken);
        ResponseEntity<MsgId> response = restTemplate.postForEntity(url, request, MsgId.class);

        // 处理错误码
        MsgId msgId = response.getBody();
        WeixinException.isSuccess(msgId);

        return msgId;
    }

    /**
     * 校验群发消息类型
     * 
     * @param message 消息
     */
    private void checkMassMsgType(@NonNull BaseRespMsg message) {

        // 校验字段是否为空
        Preconditions.checkArgument(StringUtils.isNoneBlank(message.getMsgType()), "消息类型为空");

        // 校验是否为合法的消息类型
        WxMsgType msgType = WxMsgType.get(message.getMsgType());
        Preconditions.checkArgument(msgType != null, "未知的消息类型");

        // 校验是否允许群发
        switch (msgType) {
            case MpNews:
                Preconditions.checkArgument(message instanceof MpNewsMsg, "消息类型与实例类型不匹配");
                break;
            case MpVideo:
                Preconditions.checkArgument(message instanceof MpVideoMsg, "消息类型与实例类型不匹配");
                break;
            case Text:
                Preconditions.checkArgument(message instanceof TextMsg, "消息类型与实例类型不匹配");
                break;
            case Image:
                Preconditions.checkArgument(message instanceof ImageMsg, "消息类型与实例类型不匹配");
                break;
            case Video:
                Preconditions.checkArgument(message instanceof VideoMsg, "消息类型与实例类型不匹配");
                break;
            case Voice:
                Preconditions.checkArgument(message instanceof VoiceMsg, "消息类型与实例类型不匹配");
                break;
            case WxCard:
                Preconditions.checkArgument(message instanceof WxCardMsg, "消息类型与实例类型不匹配");
                break;
            default:
                throw new IllegalArgumentException(
                    "仅支持群发下列类型的消息：文本(text), 图片(image), 图文(mpnews), 语音/音频(voice), 视频(mpvideo), 卡券(wxcard)");
        }
    }

    /**
     * 获得群发请求
     * 
     * <p>
     * 按tagId群发(message/mass/sendall)和按openIds群发(message/mass/send)时，对群发消息内容的处理方式是一样的，只是设置接收者的方式不通。因此可以将处理消息内容的方法抽取出来予以复用。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @param params 请求参数
     * @param message 消息内容
     * @return 消息体
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    private HttpEntity<?> getMassSendRequest(@NonNull String accessToken, Map<String, Object> params,
        BaseRespMsg message) throws WeixinException {

        // 校验群发消息类型
        checkMassMsgType(message);

        // 视频消息特殊处理
        // if (message instanceof VideoMsg) {
        // // 此处是应该直接自动上传，还是提示用户自己上传？
        // // 目前先按自动上传来做
        // message = convertToMpVideo(accessToken, (VideoMsg) message);
        // }

        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 根据消息类型，填充消息内容
        params.put("msgtype", message.getMsgType());
        // 填充消息内容
        params.put(message.getMsgType(), message);

        // 非原创文章是否可以转载
        if (message.getSendIgnoreReprint() != null && message.getSendIgnoreReprint() != 0) {
            params.put("send_ignore_reprint", 1);
        }

        // 使用 clientmsgid 参数，避免重复推送
        if (message.getClientMsgId() != null) {
            params.put("clientmsgid", message.getClientMsgId());
        }

        // 这里直接使用json序列化后getBytes()
        // Jackson在序列化字符串时，先把字符串转换成char[]再逐一序列化。
        // 一个emoji占两个char，会被转化成\\uXXXX\\uXXXX这种形式，在公众号就看不到emoji了。
        byte[] data = JacksonUtil.obj2Str(params).getBytes();
        return new HttpEntity<>(data, headers);
    }

    /**
     * 将普通视频消息上传，转化为mpvideo
     * 
     * @param accessToken 接口调用凭证
     * @param videoMsg 视频消息
     * @return 群发视频消息
     * @throws WeixinException
     */
    public MpVideoMsg convertToMpVideo(@NonNull String accessToken, @NonNull VideoMsg videoMsg) throws WeixinException {
        log.info("Convert {} to MpVideoMsg", videoMsg);

        // 视频消息要先上传到素材库，获得新的media_id后再群发
        MediaResult result = wxMediaAPI.uploadVideo(accessToken, videoMsg);

        // 转化为群发视频素材
        MpVideoMsg msg = new MpVideoMsg();
        msg.setMediaId(result.getMediaId());

        return msg;
    }

    /**
     * 删除群发【订阅号与服务号认证后均可用】
     * 
     * <p>
     * 群发之后，随时可以通过该接口删除群发。
     * </p>
     * 
     * 请注意：
     * 
     * <ol>
     * <li>只有已经发送成功的消息才能删除</li>
     * <li>删除消息是将消息的图文详情页失效，已经收到的用户，还是能在其本地看到消息卡片。</li>
     * <li>删除群发消息只能删除图文消息和视频消息，其他类型的消息一经发送，无法删除。</li>
     * <li>如果多次群发发送的是一个图文消息，那么删除其中一次群发，就会删除掉这个图文消息也，导致所有群发都失效</li>
     * </ol>
     * 
     * @param accessToken 接口调用凭证
     * @param params 删除群发消息的参数。
     *            <ul>
     *            <li>msgId必填，表示发送出去的消息ID；</li>
     *            <li>articleIdx选填，表示要删除的文章在图文消息中的位置，第一篇编号为1，该字段不填或填0会删除全部文章</li>
     *            </ul>
     */
    public void deleteMsg(@NonNull String accessToken, @NonNull DelMassMsg params) throws WeixinException {
        log.info("delete mass message with params={}", params);
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getMsgId()), "消息ID为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        HttpEntity<DelMassMsg> request = new HttpEntity<DelMassMsg>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassDelete, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 预览接口【订阅号与服务号认证后均可用】
     * 
     * <p>
     * 开发者可通过该接口发送消息给指定用户，在手机端查看消息的样式和排版。为了满足第三方平台开发者的需求，在保留对openID预览能力的同时，增加了对指定微信号发送预览的能力，但该能力每日调用次数有限制（100次），请勿滥用。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @param message 消息本体
     * @param openId 指定微信用户的ID
     * @param wxName 微信号
     * @return 返回消息ID
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public MsgId previewWithOpenId(@NonNull String accessToken, @NonNull BaseRespMsg message, String openId,
        String wxName) throws WeixinException {
        log.info("preview msg with openId={}, wxname={}", openId, wxName);
        Preconditions.checkArgument(StringUtils.isNoneBlank(wxName) || StringUtils.isNoneBlank("openId"),
            "openId和微信号至少有一个不能为空");

        // 请求参数
        Map<String, Object> params = Maps.newHashMap();

        // 设定消息的接收者
        if (StringUtils.isNoneBlank(wxName)) {
            params.put("towxname", wxName);
        } else {
            params.put("touser", openId);
        }

        // 设置群发消息体
        HttpEntity<?> request = getMassSendRequest(accessToken, params, message);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassPreview, accessToken);
        ResponseEntity<MsgId> response = restTemplate.postForEntity(url, request, MsgId.class);

        MsgId msgId = response.getBody();
        WeixinException.isSuccess(msgId);// 处理错误码

        return msgId;
    }

    /**
     * 查询群发消息发送状态【订阅号与服务号认证后均可用】
     * 
     * @param accessToken 接口调用凭证
     * @param msgId 群发消息后返回的消息id
     * @return WxMsgStatus，消息发送后的状态，SEND_SUCCESS表示发送成功，SENDING表示发送中，SEND_FAIL表示发送失败，DELETE表示已删除
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public WxMsgStatus getMsgStatus(@NonNull String accessToken, @NonNull String msgId) throws WeixinException {
        log.info("get mass message status with msgId={}", msgId);
        Preconditions.checkArgument(StringUtils.isNoneBlank(msgId), "消息ID为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("msg_id", msgId);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassGet, accessToken);
        ResponseEntity<MsgStatus> response = restTemplate.postForEntity(url, request, MsgStatus.class);

        MsgStatus result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        // 转化为枚举
        String msgStatus = result.getMsgStatus();
        return WxMsgStatus.valueOf(msgStatus);
    }

    /**
     * 设置群发速度
     * 
     * @param accessToken 接口调用凭证
     * @param speed 群发速度的级别，是一个0到4的整数，数字越大表示群发速度越慢。
     * 
     *            <p>
     *            speed 与 realspeed 的关系如下：
     *            </p>
     * 
     *            <table>
     *            <tr>
     *            <th>speed</th>
     *            <th>realspeed</th>
     *            </tr>
     *            <tr>
     *            <td>0</td>
     *            <td>80w/分钟</td>
     *            </tr>
     *            <tr>
     *            <td>1</td>
     *            <td>60w/分钟</td>
     *            </tr>
     *            <tr>
     *            <td>2</td>
     *            <td>45w/分钟</td>
     *            </tr>
     *            <tr>
     *            <td>3</td>
     *            <td>30w/分钟</td>
     *            </tr>
     *            <tr>
     *            <td>4</td>
     *            <td>10w/分钟</td>
     *            </tr>
     *            <tr>
     *            </tr>
     *            </table>
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void setSpeed(@NonNull String accessToken, @NonNull Integer speed) throws WeixinException {
        log.info("set mass send speed={}", speed);
        Preconditions.checkArgument(speed != null, "群发速度参数为null");
        Preconditions.checkArgument(speed >= 0 && speed <= 4, "群发速度级别应该是一个0到4的整数。");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("speed", speed);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassSetSpeed, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 获取群发速度
     * 
     * @param accessToken 接口调用凭证
     * @return 群发消息的速度
     * @see SendSpeed
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public SendSpeed getSpeed(@NonNull String accessToken) throws WeixinException {
        log.info("get mass send speed");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageMassGetSpeed, accessToken);
        // FIXME 微信文档上将这个接口描述为一个POST请求，并且不接收ACCESS_TOKEN外的任何参数。难道这不应该是个GET请求吗？
        ResponseEntity<SendSpeed> response = restTemplate.postForEntity(url, request, SendSpeed.class);

        SendSpeed result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result;
    }

}