package com.kuaike.scrm.common.service;

import com.kuaike.scrm.common.service.dto.resp.AreaDto;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 国家行政区域查询服务
 *
 * @author yanmaoyuan
 * @date 2021/9/5
 */
public interface AreaService {

    // 地区级别
    int PROVINCE = 1;// 省

    int CITY = 2;// 市

    int REGION = 3;// 区

    /**
     * 查询当前地区的级别
     *
     * @param id
     * @return
     */
    default int getLevel(String id) {
        if (id == null || id.length() != 6) {
            throw new IllegalArgumentException("invalid area id:" + id);
        }
        if (id.endsWith("0000")) {
            return PROVINCE;
        } else if (id.endsWith("00")) {
            return CITY;
        } else {
            return REGION;
        }
    }

    /**
     * 获得当前区域ID的省级行政区域ID
     *
     * @param id
     * @return
     */
    default String getProvinceId(String id) {
        if (id == null || id.length() != 6) {
            throw new IllegalArgumentException("invalid area id:" + id);
        }
        if (id.endsWith("0000")) {
            return id;
        } else {
            return id.substring(0, 2) + "0000";
        }
    }

    /**
     * 获得当前区域ID的市级行政区域ID
     *
     * @param id
     * @return
     */
    default String getCityId(String id) {
        if (id == null || id.length() != 6) {
            throw new IllegalArgumentException("invalid area id:" + id);
        }
        if (id.endsWith("0000")) {
            throw new IllegalArgumentException("Can't get parent city of a province:" + id);
        } else if (id.endsWith("00")) {
            return id;
        } else {
            return id.substring(0, 4) + "00";
        }
    }

    /**
     * 获取指定区域的名称
     *
     * eg: "420111" -> "洪山区"
     *
     * @param id 行政区域编码
     * @return 地区名称
     */
    String getName(String id);

    /**
     * 批量查询区域的名称
     *
     * @param ids
     * @return
     */
    Map<String, String> getNames(Collection<String> ids);

    /**
     * 获得指定区域的完整名称
     *
     * eg: "420111" -> "湖北省-武汉市-洪山区"
     *
     * @param id
     * @return
     */
    String getFullName(String id);

    /**
     * 批量查询区域的完整名称
     *
     * @param ids
     * @return
     */
    Map<String, String> getFullNames(Collection<String> ids);

    /**
     * 查询指定区域的所有下级区域编码，递归
     *
     * @param id
     * @param containSelf 是否包含自身
     * @return
     */
    List<String> getChildrenIds(String id, boolean containSelf);

    /**
     * 查询指定区域的所有直接下级区域编码，不递归
     *
     * @param id
     * @param containSelf 是否包含自身
     * @return
     */
    List<String> getDirectChildrenIds(String id, boolean containSelf);

    /**
     * 查询所有省级区域
     *
     * @return
     */
    List<AreaDto> getProvince();

    /**
     * 根据区域编码，查询直接下级区域（不递归）
     *
     * @param id
     * @return
     */
    List<AreaDto> getChildren(String id);
}
