package com.kuaike.scrm.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

/**
 * @author louis
 * @date 2022/5/25
 */
@Slf4j
@SuppressWarnings("all")
public class TraceIdUtils {

    public static enum TraceType{
        KAFKA,SCRM,JOB,CRM,TRADE,AD,CUSTOMER,PAY,ORDER
    }

    private TraceIdUtils() {
    }
    /**
     * 日志配置文件中有一个traceId，这两者保持一致
     */

    public static final String TRACE_ID = "traceId";

    /**
     * 设置traceId
     */
    public static void setTraceId(String traceId) {
        if (StringUtils.isNotBlank(traceId)) {
            //将traceId放到MDC中
            MDC.put(TRACE_ID, traceId);
        }
    }
    /**
     * 获取traceId
     */
    public static String getTraceId() {
        //获取
        return MDC.get(TRACE_ID);

    }
    /**
     * 生成traceId
     */
    public static String generateTraceId(TraceType traceType) {
        String type = Objects.isNull(traceType) ? "" : traceType.toString();
        String originTraceId = UUID.randomUUID().toString().replace("-", "");
        String traceId = null;
        if (StringUtils.isNotBlank(type)) {
            traceId = type + "-";
        }
        traceId = traceId + originTraceId.substring(0, 20);

        return traceId;
    }

    public static String genAndSetTraceId(TraceType traceType) {
        String traceId = generateTraceId(traceType);
        setTraceId(traceId);
        return traceId;
    }
    public static void removeTraceId() {
        MDC.remove(TRACE_ID);
    }

    public static Map<String,String> getMDCContactMap() {
       return MDC.getCopyOfContextMap();
    }

    /**
     * 异步情况设置全局traceIdMap信息
     * 如果有异步操作，请配合 {@link TraceIdUtils#removeTraceId()} 使用
     *
     * @param contextMap
     *
     */
    public static void setContextMap(Map<String, String> contextMap) {
        if (MapUtils.isNotEmpty(contextMap)) {
            MDC.setContextMap(contextMap);
        }

    }

    public static void generateTraceIdToRequest(HttpServletRequest request, TraceType traceType) {
        try {
            String traceId;
            if (Objects.nonNull(request.getAttribute(TraceIdUtils.TRACE_ID))) {
                traceId = request.getAttribute(TraceIdUtils.TRACE_ID).toString();
                TraceIdUtils.setTraceId(traceId);
            } else {
                traceId = TraceIdUtils.genAndSetTraceId(traceType);
            }
            request.setAttribute(TraceIdUtils.TRACE_ID, traceId);
        } catch (Exception e) {
            String requestURI = request.getRequestURI();
            log.error("生成traceId异常:urI:{}",requestURI, e);
        }
    }

    public static void generateTraceIdToRequest(HttpServletRequest request) {
        generateTraceIdToRequest(request, TraceType.SCRM);
    }


}
