/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.multimap.bag.strategy;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.block.HashingStrategy;
import com.gs.collections.api.block.function.Function;
import com.gs.collections.api.block.predicate.Predicate2;
import com.gs.collections.api.map.MutableMap;
import com.gs.collections.api.multimap.Multimap;
import com.gs.collections.api.multimap.bag.MutableBagMultimap;
import com.gs.collections.api.tuple.Pair;
import com.gs.collections.impl.bag.mutable.HashBag;
import com.gs.collections.impl.map.strategy.mutable.UnifiedMapWithHashingStrategy;
import com.gs.collections.impl.multimap.bag.AbstractMutableBagMultimap;
import com.gs.collections.impl.utility.Iterate;

public final class HashBagMultimapWithHashingStrategy<K, V>
        extends AbstractMutableBagMultimap<K, V> implements Externalizable
{
    private static final long serialVersionUID = 1L;
    private HashingStrategy<? super K> hashingStrategy;

    /**
     * @deprecated Empty default constructor used for serialization. Instantiating an HashBagMultimapWithHashingStrategy with
     * this constructor will have a null multimapHashingStrategy, and throw NullPointerException when used.
     */
    @SuppressWarnings("UnusedDeclaration")
    @Deprecated
    public HashBagMultimapWithHashingStrategy()
    {
        // For Externalizable use only
    }

    public HashBagMultimapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy)
    {
        this.hashingStrategy = hashingStrategy;
        this.map = this.createMap();
    }

    public HashBagMultimapWithHashingStrategy(HashBagMultimapWithHashingStrategy<K, V> multimap)
    {
        this(multimap.hashingStrategy, multimap);
    }

    public HashBagMultimapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy, Multimap<? extends K, ? extends V> multimap)
    {
        this.hashingStrategy = hashingStrategy;
        this.map = this.createMapWithKeyCount(Math.max(multimap.sizeDistinct() * 2, 16));
        this.putAll(multimap);
    }

    public HashBagMultimapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy, Pair<K, V>... pairs)
    {
        this(hashingStrategy);
        this.putAllPairs(pairs);
    }

    public HashBagMultimapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy, Iterable<Pair<K, V>> inputIterable)
    {
        this(hashingStrategy);
        for (Pair<K, V> single : inputIterable)
        {
            this.add(single);
        }
    }

    public static <K, V> HashBagMultimapWithHashingStrategy<K, V> newMultimap(HashBagMultimapWithHashingStrategy<K, V> multimap)
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(multimap);
    }

    public static <K, V> HashBagMultimapWithHashingStrategy<K, V> newMultimap(HashingStrategy<? super K> multimapHashingStrategy, Multimap<? extends K, ? extends V> multimap)
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(multimapHashingStrategy, multimap);
    }

    public static <K, V> HashBagMultimapWithHashingStrategy<K, V> newMultimap(HashingStrategy<? super K> multimapHashingStrategy)
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(multimapHashingStrategy);
    }

    @SafeVarargs
    public static <K, V> HashBagMultimapWithHashingStrategy<K, V> newMultimap(HashingStrategy<? super K> multimapHashingStrategy, Pair<K, V>... pairs)
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(multimapHashingStrategy, pairs);
    }

    public static <K, V> HashBagMultimapWithHashingStrategy<K, V> newMultimap(HashingStrategy<? super K> multimapHashingStrategy, Iterable<Pair<K, V>> inputIterable)
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(multimapHashingStrategy, inputIterable);
    }

    @Override
    protected MutableMap<K, MutableBag<V>> createMap()
    {
        return UnifiedMapWithHashingStrategy.newMap(this.hashingStrategy);
    }

    @Override
    protected MutableMap<K, MutableBag<V>> createMapWithKeyCount(int keyCount)
    {
        return UnifiedMapWithHashingStrategy.newMap(this.hashingStrategy, keyCount);
    }

    @Override
    protected MutableBag<V> createCollection()
    {
        return HashBag.newBag();
    }

    public HashingStrategy<? super K> getKeyHashingStrategy()
    {
        return this.hashingStrategy;
    }

    public HashBagMultimapWithHashingStrategy<K, V> newEmpty()
    {
        return new HashBagMultimapWithHashingStrategy<K, V>(this.hashingStrategy);
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeObject(this.hashingStrategy);
        super.writeExternal(out);
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        this.hashingStrategy = (HashingStrategy<? super K>) in.readObject();
        super.readExternal(in);
    }

    // Currently this returns a HashBagMultimap.
    // On a future release, it will return HashBagWithHashingStrategyMultimap, where the HashBag collection hashing strategy
    // will be the hashing strategy of this multimap
    public MutableBagMultimap<V, K> flip()
    {
        return Iterate.flip(this);
    }

    public <V2> HashBagMultimapWithHashingStrategy<K, V2> collectValues(Function<? super V, ? extends V2> function)
    {
        return this.collectValues(function, HashBagMultimapWithHashingStrategy.<K, V2>newMultimap(this.hashingStrategy));
    }

    public HashBagMultimapWithHashingStrategy<K, V> selectKeysValues(Predicate2<? super K, ? super V> predicate)
    {
        return this.selectKeysValues(predicate, this.newEmpty());
    }

    public HashBagMultimapWithHashingStrategy<K, V> rejectKeysValues(Predicate2<? super K, ? super V> predicate)
    {
        return this.rejectKeysValues(predicate, this.newEmpty());
    }

    public HashBagMultimapWithHashingStrategy<K, V> selectKeysMultiValues(Predicate2<? super K, ? super Iterable<V>> predicate)
    {
        return this.selectKeysMultiValues(predicate, this.newEmpty());
    }

    public HashBagMultimapWithHashingStrategy<K, V> rejectKeysMultiValues(Predicate2<? super K, ? super Iterable<V>> predicate)
    {
        return this.rejectKeysMultiValues(predicate, this.newEmpty());
    }
}
