/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.RichIterable;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.block.HashingStrategy;
import com.gs.collections.api.block.function.primitive.LongFunction;
import com.gs.collections.api.block.function.primitive.LongFunction0;
import com.gs.collections.api.block.function.primitive.LongToLongFunction;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.predicate.primitive.ObjectLongPredicate;
import com.gs.collections.api.block.procedure.Procedure;
import com.gs.collections.api.block.procedure.Procedure2;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.block.procedure.primitive.ObjectLongProcedure;
import com.gs.collections.api.collection.MutableCollection;
import com.gs.collections.api.collection.primitive.ImmutableLongCollection;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.iterator.MutableLongIterator;
import com.gs.collections.api.list.MutableList;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.map.primitive.ImmutableObjectLongMap;
import com.gs.collections.api.map.primitive.MutableObjectLongMap;
import com.gs.collections.api.map.primitive.ObjectLongMap;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.api.tuple.primitive.ObjectLongPair;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.collection.mutable.primitive.SynchronizedLongCollection;
import com.gs.collections.impl.collection.mutable.primitive.UnmodifiableLongCollection;
import com.gs.collections.impl.lazy.AbstractLazyIterable;
import com.gs.collections.impl.factory.primitive.LongLists;
import com.gs.collections.impl.factory.primitive.ObjectLongMaps;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.list.mutable.FastList;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.api.block.procedure.primitive.ObjectIntProcedure;
import com.gs.collections.impl.tuple.primitive.PrimitiveTuples;
/**
 * This file was automatically generated from template file objectPrimitiveHashMapWithHashingStrategy.stg.
 *
 * @since 7.0.
 */
public class ObjectLongHashMapWithHashingStrategy<K> implements MutableObjectLongMap<K>, Externalizable
{
    public static final long EMPTY_VALUE = 0L;

    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private static final Object NULL_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectLongHashMapWithHashingStrategy.NULL_KEY";
        }
    };

    private static final Object REMOVED_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectLongHashMapWithHashingStrategy.REMOVED_KEY";
        }
    };

    private Object[] keys;
    private long[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    private HashingStrategy<? super K> hashingStrategy;

    @Deprecated
    public ObjectLongHashMapWithHashingStrategy()
    {
    }

    public ObjectLongHashMapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy)
    {
        this.hashingStrategy = hashingStrategy;
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ObjectLongHashMapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy, int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        this.hashingStrategy = hashingStrategy;
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    public ObjectLongHashMapWithHashingStrategy(HashingStrategy<? super K> hashingStrategy, ObjectLongMap<? extends K> map)
    {
        this(hashingStrategy, Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newMap(HashingStrategy<? super K> hashingStrategy)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newMap(HashingStrategy<? super K> hashingStrategy, ObjectLongMap<K> map)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy, map);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newMap(ObjectLongHashMapWithHashingStrategy<K> map)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(map.hashingStrategy, map);
    }


    public static <K> ObjectLongHashMapWithHashingStrategy<K> newWithKeysValues(HashingStrategy<? super K> hashingStrategy, K key1, long value1)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy, 1).withKeyValue(key1, value1);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newWithKeysValues(HashingStrategy<? super K> hashingStrategy, K key1, long value1, K key2, long value2)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy, 2).withKeysValues(key1, value1, key2, value2);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newWithKeysValues(HashingStrategy<? super K> hashingStrategy, K key1, long value1, K key2, long value2, K key3, long value3)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy, 3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static <K> ObjectLongHashMapWithHashingStrategy<K> newWithKeysValues(HashingStrategy<? super K> hashingStrategy, K key1, long value1, K key2, long value2, K key3, long value3, K key4, long value4)
    {
        return new ObjectLongHashMapWithHashingStrategy<K>(hashingStrategy, 4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ObjectLongMap))
        {
            return false;
        }

        ObjectLongMap<K> other = (ObjectLongMap<K>) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!other.containsKey(this.toNonSentinel(this.keys[i])) || this.values[i] != other.getOrThrow(this.toNonSentinel(this.keys[i]))))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.hashingStrategy.computeHashCode(this.toNonSentinel(this.keys[i])) ^ (int) (this.values[i] ^ this.values[i] >>> 32);
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        StringBuilder appendable = new StringBuilder();

        appendable.append("{");

        boolean first = true;

        for (int i = 0; i < this.keys.length; i++)
        {
            Object key = this.keys[i];
            if (isNonSentinel(key))
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(this.toNonSentinel(key)).append("=").append(this.values[i]);
                first = false;
            }
        }
        appendable.append("}");

        return appendable.toString();
    }

    public int size()
    {
        return this.occupiedWithData;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            for (int i = 0; i < this.keys.length; i++)
            {
                Object key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(String.valueOf(this.values[i])));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableLongIterator longIterator()
    {
        return new InternalLongIterator();
    }

    public long[] toArray()
    {
        long[] result = new long[this.size()];
        int index = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result[index] = this.values[i];
                index++;
            }
        }
        return result;
    }

    public boolean contains(long value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(long... source)
    {
        for (long item : source)
        {
            if (!this.containsValue(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        return this.containsAll(source.toArray());
    }

    public void clear()
    {
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, null);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(K key, long value)
    {
        int index = this.probe(key);

        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        this.addKeyValueAtIndex(key, value, index);
    }

    public void putAll(ObjectLongMap<? extends K> map)
    {
        map.forEachKeyValue(new ObjectLongProcedure<K>()
        {
            public void value(K key, long value)
            {
                ObjectLongHashMapWithHashingStrategy.this.put(key, value);
            }
        });
    }

    public void removeKey(K key)
    {
        int index = this.probe(key);
        this.removeKeyAtIndex(key, index);
    }

    public void removeKeyAtIndex(K key, int index)
    {
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
        }
    }

    public void remove(Object key)
    {
        this.removeKey((K) key);
    }

    public long removeKeyIfAbsent(K key, long value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            long oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;

            return oldValue;
        }
        return value;
    }

    public long getIfAbsentPut(K key, long value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPut(K key, LongFunction0 function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        long value = function.value();
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public <P> long getIfAbsentPutWith(K key, LongFunction<? super P> function, P parameter)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        long value = function.longValueOf(parameter);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long getIfAbsentPutWithKey(K key, LongFunction<? super K> function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        long value = function.longValueOf(key);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public long updateValue(K key, long initialValueIfAbsent, LongToLongFunction function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        long value = function.valueOf(initialValueIfAbsent);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    private void addKeyValueAtIndex(K key, long value, int index)
    {
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData + this.occupiedWithSentinels > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public long addToValue(K key, long toBeAdded)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] += toBeAdded;
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, toBeAdded, index);
        return this.values[index];
    }

    public ObjectLongHashMapWithHashingStrategy<K> withKeyValue(K key1, long value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ObjectLongHashMapWithHashingStrategy<K> withKeysValues(K key1, long value1, K key2, long value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ObjectLongHashMapWithHashingStrategy<K> withKeysValues(K key1, long value1, K key2, long value2, K key3, long value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ObjectLongHashMapWithHashingStrategy<K> withKeysValues(K key1, long value1, K key2, long value2, K key3, long value3, K key4, long value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ObjectLongHashMapWithHashingStrategy<K> withoutKey(K key)
    {
        this.removeKey(key);
        return this;
    }

    public ObjectLongHashMapWithHashingStrategy<K> withoutAllKeys(Iterable<? extends K> keys)
    {
        for (K key : keys)
        {
            this.removeKey(key);
        }
        return this;
    }

    public MutableObjectLongMap<K> asUnmodifiable()
    {
        return new UnmodifiableObjectLongMap<K>(this);
    }

    public MutableObjectLongMap<K> asSynchronized()
    {
        return new SynchronizedObjectLongMap<K>(this);
    }

    public ImmutableObjectLongMap<K> toImmutable()
    {
        return ObjectLongMaps.immutable.withAll(this);
    }

    public long get(Object key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public long getOrThrow(Object key)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public long getIfAbsent(Object key, long ifAbsent)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public boolean containsKey(Object key)
    {
        int index = this.probe(key);
        return isNonSentinel(this.keys[index]) && this.nullSafeEquals(this.toNonSentinel(this.keys[index]), key);
    }

    public boolean containsValue(long value)
    {
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && this.values[i] == value)
            {
                return true;
            }
        }
        return false;
    }

    public void forEach(LongProcedure procedure)
    {
        this.each(procedure);
    }

    public void each(LongProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(LongProcedure procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(Procedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]));
            }
        }
    }

    public void forEachKeyValue(ObjectLongProcedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
    }

    public ObjectLongHashMapWithHashingStrategy<K> select(ObjectLongPredicate<? super K> predicate)
    {
        ObjectLongHashMapWithHashingStrategy<K> result = ObjectLongHashMapWithHashingStrategy.newMap(this.hashingStrategy);

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public ObjectLongHashMapWithHashingStrategy<K> reject(ObjectLongPredicate<? super K> predicate)
    {
        ObjectLongHashMapWithHashingStrategy<K> result = ObjectLongHashMapWithHashingStrategy.newMap(this.hashingStrategy);

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public MutableLongCollection select(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableLongCollection reject(LongPredicate predicate)
    {
        LongArrayList result = new LongArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public long detectIfNone(LongPredicate predicate, long ifNone)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return ifNone;
    }

    public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
    {
        MutableList<V> result = FastList.newList(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(function.valueOf(this.values[i]));
            }
        }
        return result;
    }

    public int count(LongPredicate predicate)
    {
        int count = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectLongToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public long sum()
    {
        long result = 0L;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += this.values[i];
            }
        }
        return result;
    }

    public long max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        long max = 0L;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public long min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        long min = 0L;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public long maxIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        long max = 0L;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || max < this.values[i]))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public long minIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        long min = 0L;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || this.values[i] < min))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public MutableLongList toList()
    {
        MutableLongList result = new LongArrayList(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableLongSet toSet()
    {
        MutableLongSet result = new LongHashSet(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableLongBag toBag()
    {
        MutableLongBag result = new LongHashBag(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableLongList toSortedList()
    {
        return this.toList().sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeObject(this.hashingStrategy);
        out.writeInt(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeObject(this.toNonSentinel(this.keys[i]));
                out.writeLong(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        this.hashingStrategy = (HashingStrategy<? super K>) in.readObject();
        int size = in.readInt();
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(size * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);

        for (int i = 0; i < size; i++)
        {
            this.put((K) in.readObject(), in.readLong());
        }
    }

    public LazyIterable<K> keysView()
    {
        return new KeysView();
    }

    public RichIterable<ObjectLongPair<K>> keyValuesView()
    {
        return new KeyValuesView();
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        Object[] old = this.keys;
        long[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(this.toNonSentinel(old[i]), oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(Object element)
    {
        int index = this.spread(element);

        int removedIndex = -1;
        if (isRemovedKey(this.keys[index]))
        {
            removedIndex = index;
        }

        else if (this.keys[index] == null || this.nullSafeEquals(this.toNonSentinel(this.keys[index]), element))
        {
            return index;
        }

        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = no. of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (isRemovedKey(this.keys[nextIndex]))
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (this.nullSafeEquals(this.toNonSentinel(this.keys[nextIndex]), element))
            {
                return nextIndex;
            }
            else if (this.keys[nextIndex] == null)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(Object element)
    {
        // This function ensures that hashCodes that differ only by
        // constant multiples at each bit position have a bounded
        // number of collisions (approximately 8 at default load factor).
        K nonSentinelKey = this.toNonSentinel(element);
        int h = nonSentinelKey == null ? 0 : this.hashingStrategy.computeHashCode(nonSentinelKey);
        h ^= h >>> 20 ^ h >>> 12;
        h ^= h >>> 7 ^ h >>> 4;
        return h & (this.keys.length - 1);
    }

    private boolean nullSafeEquals(K value, Object other)
    {
        if (value == null && other == null)
        {
            return true;
        }
        else if (value != NULL_KEY && other != null)
        {
            if (this.hashingStrategy.equals(value, this.toNonSentinel(other)))
            {
                return true;
            }
        }
        return false;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new Object[sizeToAllocate];
        this.values = new long[sizeToAllocate];
    }

    private static boolean isRemovedKey(Object key)
    {
        return key == REMOVED_KEY;
    }

    private static <K> boolean isNonSentinel(K key)
    {
        return key != null && !isRemovedKey(key);
    }

    private K toNonSentinel(Object key)
    {
        return key == NULL_KEY ? null : (K) key;
    }

    private static Object toSentinelIfNull(Object key)
    {
        return key == null ? NULL_KEY : key;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private class InternalLongIterator implements MutableLongIterator
    {
        private int count;
        private int position;

        public boolean hasNext()
        {
            return this.count != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }

            Object[] keys = ObjectLongHashMapWithHashingStrategy.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            long result = ObjectLongHashMapWithHashingStrategy.this.values[this.position];
            this.count++;
            this.position++;
            return result;
        }

        public void remove()
        {
            if (this.position == 0 || !isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[this.position - 1]))
            {
                throw new IllegalStateException();
            }
            ObjectLongHashMapWithHashingStrategy.this.remove(ObjectLongHashMapWithHashingStrategy.this.keys[this.position - 1]);
            this.count--;
        }
    }

    public Set<K> keySet()
    {
        return new KeySet();
    }

    public MutableLongCollection values()
    {
        return new ValuesCollection();
    }

    private class KeySet implements Set<K>
    {
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof Set)
            {
                Set<?> other = (Set<?>) obj;
                if (other.size() == this.size())
                {
                    return this.containsAll(other);
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            int hashCode = 0;
            Object[] table = ObjectLongHashMapWithHashingStrategy.this.keys;
            for (int i = 0; i < table.length; i++)
            {
                Object key = table[i];
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(key))
                {
                    K nonSentinelKey = ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(key);
                    hashCode += nonSentinelKey == null ? 0 : ObjectLongHashMapWithHashingStrategy.this.hashingStrategy.computeHashCode(nonSentinelKey);
                }
            }
            return hashCode;
        }

        public int size()
        {
            return ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectLongHashMapWithHashingStrategy.this.isEmpty();
        }

        public boolean contains(Object o)
        {
            return ObjectLongHashMapWithHashingStrategy.this.containsKey(o);
        }

        public Object[] toArray()
        {
            int size = ObjectLongHashMapWithHashingStrategy.this.size();
            Object[] result = new Object[size];
            this.copyKeys(result);
            return result;
        }

        public <T> T[] toArray(T[] result)
        {
            int size = ObjectLongHashMapWithHashingStrategy.this.size();
            if (result.length < size)
            {
                result = (T[]) Array.newInstance(result.getClass().getComponentType(), size);
            }
            this.copyKeys(result);
            if (size < result.length)
            {
                result[size] = null;
            }
            return result;
        }

        public boolean add(K key)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean remove(Object key)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();
            ObjectLongHashMapWithHashingStrategy.this.removeKey((K) key);
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean containsAll(Collection<?> collection)
        {
            for (Object aCollection : collection)
            {
                if (!ObjectLongHashMapWithHashingStrategy.this.containsKey(aCollection))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean addAll(Collection<? extends K> collection)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean retainAll(Collection<?> collection)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();
            Iterator<K> iterator = this.iterator();
            while (iterator.hasNext())
            {
                K next = iterator.next();
                if (!collection.contains(next))
                {
                    iterator.remove();
                }
            }
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean removeAll(Collection<?> collection)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();
            for (Object object : collection)
            {
                ObjectLongHashMapWithHashingStrategy.this.removeKey((K) object);
            }
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public void clear()
        {
            ObjectLongHashMapWithHashingStrategy.this.clear();
        }

        public Iterator<K> iterator()
        {
            return new KeySetIterator();
        }

        private void copyKeys(Object[] result)
        {
            int count = 0;
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                Object key = ObjectLongHashMapWithHashingStrategy.this.keys[i];
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(key))
                {
                    result[count++] = ObjectLongHashMapWithHashingStrategy.this.keys[i];
                }
            }
        }
    }

    private class KeySetIterator implements Iterator<K>
    {
        private int count;
        private int position;
        private K currentKey;
        private boolean isCurrentKeySet;

        public boolean hasNext()
        {
            return this.count < ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public K next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            this.count++;
            Object[] keys = ObjectLongHashMapWithHashingStrategy.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            this.currentKey = (K) ObjectLongHashMapWithHashingStrategy.this.keys[this.position];
            this.isCurrentKeySet = true;
            this.position++;
            return ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(this.currentKey);
        }

        public void remove()
        {
            if (!this.isCurrentKeySet)
            {
                throw new IllegalStateException();
            }

            this.isCurrentKeySet = false;
            this.count--;

            if (isNonSentinel(this.currentKey))
            {
                int index = this.position - 1;
                ObjectLongHashMapWithHashingStrategy.this.removeKeyAtIndex(this.currentKey, index);
            }
            else
            {
                ObjectLongHashMapWithHashingStrategy.this.removeKey(this.currentKey);
            }
        }
    }

    private class ValuesCollection implements MutableLongCollection
    {
        public int size()
        {
            return ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectLongHashMapWithHashingStrategy.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ObjectLongHashMapWithHashingStrategy.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
                {
                    Object key = ObjectLongHashMapWithHashingStrategy.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ObjectLongHashMapWithHashingStrategy.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public boolean add(long element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(long... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(LongIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(long item)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();

            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]) && item == ObjectLongHashMapWithHashingStrategy.this.values[i])
                {
                    ObjectLongHashMapWithHashingStrategy.this.removeKey((K) ObjectLongHashMapWithHashingStrategy.this.keys[i]);
                }
            }
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean removeAll(LongIterable source)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();

            LongIterator iterator = source.longIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean removeAll(long... source)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();

            for (long item : source)
            {
                this.remove(item);
            }
            return oldSize != ObjectLongHashMapWithHashingStrategy.this.size();
        }

        public boolean retainAll(LongIterable source)
        {
            int oldSize = ObjectLongHashMapWithHashingStrategy.this.size();
            final LongSet sourceSet = source instanceof LongSet ? (LongSet) source : source.toSet();
            ObjectLongHashMapWithHashingStrategy<K> retained = ObjectLongHashMapWithHashingStrategy.this.select(new ObjectLongPredicate<K>()
            {
                public boolean accept(K object, long value)
                {
                    return sourceSet.contains(value);
                }
            });
            if (retained.size() != oldSize)
            {
                ObjectLongHashMapWithHashingStrategy.this.keys = retained.keys;
                ObjectLongHashMapWithHashingStrategy.this.values = retained.values;
                ObjectLongHashMapWithHashingStrategy.this.occupiedWithData = retained.occupiedWithData;
                ObjectLongHashMapWithHashingStrategy.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(long... source)
        {
            return this.retainAll(LongHashSet.newSetWith(source));
        }

        public void clear()
        {
            ObjectLongHashMapWithHashingStrategy.this.clear();
        }

        public MutableLongCollection with(long element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection without(long element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection withoutAll(LongIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public MutableLongCollection asUnmodifiable()
        {
            return UnmodifiableLongCollection.of(this);
        }

        public MutableLongCollection asSynchronized()
        {
            return SynchronizedLongCollection.of(this);
        }

        public ImmutableLongCollection toImmutable()
        {
            return LongLists.immutable.withAll(this);
        }

        public MutableLongIterator longIterator()
        {
            return ObjectLongHashMapWithHashingStrategy.this.longIterator();
        }

        public long[] toArray()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toArray();
        }

        public boolean contains(long value)
        {
            return ObjectLongHashMapWithHashingStrategy.this.containsValue(value);
        }

        public boolean containsAll(long... source)
        {
            return ObjectLongHashMapWithHashingStrategy.this.containsAll(source);
        }

        public boolean containsAll(LongIterable source)
        {
            return ObjectLongHashMapWithHashingStrategy.this.containsAll(source);
        }

        public void forEach(LongProcedure procedure)
        {
            ObjectLongHashMapWithHashingStrategy.this.forEach(procedure);
        }

        public void each(LongProcedure procedure)
        {
            this.forEach(procedure);
        }

        public MutableLongCollection select(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.select(predicate);
        }

        public MutableLongCollection reject(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.reject(predicate);
        }

        public <V> MutableCollection<V> collect(LongToObjectFunction<? extends V> function)
        {
            return ObjectLongHashMapWithHashingStrategy.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
        {
            return ObjectLongHashMapWithHashingStrategy.this.injectInto(injectedValue, function);
        }

        public long detectIfNone(LongPredicate predicate, long ifNone)
        {
            return ObjectLongHashMapWithHashingStrategy.this.detectIfNone(predicate, ifNone);
        }

        public int count(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.count(predicate);
        }

        public boolean anySatisfy(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(LongPredicate predicate)
        {
            return ObjectLongHashMapWithHashingStrategy.this.noneSatisfy(predicate);
        }

        public MutableLongList toList()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toList();
        }

        public MutableLongSet toSet()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toSet();
        }

        public MutableLongBag toBag()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toBag();
        }

        public LazyLongIterable asLazy()
        {
            return new LazyLongIterableAdapter(this);
        }

        public long[] toSortedArray()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toSortedArray();
        }

        public MutableLongList toSortedList()
        {
            return ObjectLongHashMapWithHashingStrategy.this.toSortedList();
        }

        public long sum()
        {
            return ObjectLongHashMapWithHashingStrategy.this.sum();
        }

        public long max()
        {
            return ObjectLongHashMapWithHashingStrategy.this.max();
        }

        public long maxIfEmpty(long defaultValue)
        {
            return ObjectLongHashMapWithHashingStrategy.this.maxIfEmpty(defaultValue);
        }

        public long min()
        {
            return ObjectLongHashMapWithHashingStrategy.this.min();
        }

        public long minIfEmpty(long defaultValue)
        {
            return ObjectLongHashMapWithHashingStrategy.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ObjectLongHashMapWithHashingStrategy.this.average();
        }

        public double median()
        {
            return ObjectLongHashMapWithHashingStrategy.this.median();
        }
    }

    private class KeysView extends AbstractLazyIterable<K>
    {
        public void each(Procedure<? super K> procedure)
        {
            ObjectLongHashMapWithHashingStrategy.this.forEachKey(procedure);
        }

        public void forEachWithIndex(ObjectIntProcedure<? super K> objectLongProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]))
                {
                    objectLongProcedure.value(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super K, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]))
                {
                    procedure.value(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]), parameter);
                }
            }
        }

        public Iterator<K> iterator()
        {
            return new InternalKeysViewIterator();
        }

        public class InternalKeysViewIterator implements Iterator<K>
        {
            private int count;
            private int position;

            public K next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectLongHashMapWithHashingStrategy.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                K result = ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectLongHashMapWithHashingStrategy.this.size();
            }
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<ObjectLongPair<K>>
    {
        public void each(Procedure<? super ObjectLongPair<K>> procedure)
        {
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]), ObjectLongHashMapWithHashingStrategy.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super ObjectLongPair<K>> objectIntProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]), ObjectLongHashMapWithHashingStrategy.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super ObjectLongPair<K>, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectLongHashMapWithHashingStrategy.this.keys.length; i++)
            {
                if (ObjectLongHashMapWithHashingStrategy.isNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[i]), ObjectLongHashMapWithHashingStrategy.this.values[i]), parameter);
                }
            }
        }

        public Iterator<ObjectLongPair<K>> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<ObjectLongPair<K>>
        {
            private int count;
            private int position;

            public ObjectLongPair<K> next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectLongHashMapWithHashingStrategy.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                ObjectLongPair<K> result = PrimitiveTuples.pair(ObjectLongHashMapWithHashingStrategy.this.toNonSentinel(ObjectLongHashMapWithHashingStrategy.this.keys[this.position]), ObjectLongHashMapWithHashingStrategy.this.values[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectLongHashMapWithHashingStrategy.this.size();
            }
        }
    }
}
