/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.block.factory.primitive;

import com.gs.collections.api.block.predicate.primitive.DoublePredicate;

/**
 * Provides a set of common predicates for double values.
 * This file was automatically generated from template file primitivePredicates.stg.
 */
public final class DoublePredicates
{
    private static final DoublePredicate ALWAYS_TRUE = new AlwaysTrueDoublePredicate();
    private static final DoublePredicate ALWAYS_FALSE = new AlwaysFalseDoublePredicate();

    private DoublePredicates()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    @SuppressWarnings("MisspelledEquals")
    public static DoublePredicate equal(double expected)
    {
        return new EqualsDoublePredicate(expected);
    }

    public static DoublePredicate equal(double expected, double delta)
    {
        return new EqualsWithDeltaDoublePredicate(expected, delta);
    }

    public static DoublePredicate lessThan(double expected)
    {
        return new LessThanDoublePredicate(expected);
    }

    public static DoublePredicate greaterThan(double expected)
    {
        return new GreaterThanDoublePredicate(expected);
    }

    public static DoublePredicate alwaysTrue()
    {
        return ALWAYS_TRUE;
    }

    public static DoublePredicate alwaysFalse()
    {
        return ALWAYS_FALSE;
    }

    public static DoublePredicate and(DoublePredicate one, DoublePredicate two)
    {
        return new AndDoublePredicate(one, two);
    }

    public static DoublePredicate or(DoublePredicate one, DoublePredicate two)
    {
        return new OrDoublePredicate(one, two);
    }

    public static DoublePredicate not(DoublePredicate negate)
    {
        return new NotDoublePredicate(negate);
    }

    private static final class EqualsDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;
        private final double expected;

        private EqualsDoublePredicate(double expected)
        {
            this.expected = expected;
        }

        public boolean accept(double actual)
        {
            return Double.compare(actual, this.expected) == 0;
        }
    }

    private static final class EqualsWithDeltaDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;
        private final double expected;
        private final double delta;

        private EqualsWithDeltaDoublePredicate(double expected, double delta)
        {
            this.expected = expected;
            this.delta = delta;
        }

        public boolean accept(double actual)
        {
            return Math.abs(this.expected - actual) <= this.delta;
        }
    }

    private static final class LessThanDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        private final double expected;

        private LessThanDoublePredicate(double expected)
        {
            this.expected = expected;
        }

        public boolean accept(double actual)
        {
            return actual < this.expected;
        }
    }

    private static final class GreaterThanDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        private final double expected;

        private GreaterThanDoublePredicate(double expected)
        {
            this.expected = expected;
        }

        public boolean accept(double actual)
        {
            return actual > this.expected;
        }
    }

    private static final class AndDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        private final DoublePredicate one;
        private final DoublePredicate two;

        private AndDoublePredicate(DoublePredicate one, DoublePredicate two)
        {
            this.one = one;
            this.two = two;
        }

        public boolean accept(double actual)
        {
            return this.one.accept(actual) && this.two.accept(actual);
        }
    }

    private static final class OrDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        private final DoublePredicate one;
        private final DoublePredicate two;

        private OrDoublePredicate(DoublePredicate one, DoublePredicate two)
        {
            this.one = one;
            this.two = two;
        }

        public boolean accept(double actual)
        {
            return this.one.accept(actual) || this.two.accept(actual);
        }
    }

    private static final class NotDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        private final DoublePredicate negate;

        private NotDoublePredicate(DoublePredicate negate)
        {
            this.negate = negate;
        }

        public boolean accept(double actual)
        {
            return !this.negate.accept(actual);
        }
    }


    private static final class AlwaysTrueDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(double value)
        {
            return true;
        }
    }

    private static final class AlwaysFalseDoublePredicate implements DoublePredicate
    {
        private static final long serialVersionUID = 1L;

        public boolean accept(double value)
        {
            return false;
        }
    }
}
