/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableDoubleBag;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleIntProcedure;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedDoubleCollection;
import com.gs.collections.impl.factory.primitive.DoubleBags;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableDoubleBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link DoubleIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 *
 * @see MutableDoubleBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedDoubleBag
        extends AbstractSynchronizedDoubleCollection
        implements MutableDoubleBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedDoubleBag(MutableDoubleBag bag)
    {
        super(bag);
    }

    SynchronizedDoubleBag(MutableDoubleBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableDoubleBag getMutableDoubleBag()
    {
        return (MutableDoubleBag) this.getDoubleCollection();
    }

    @Override
    public SynchronizedDoubleBag with(double element)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedDoubleBag without(double element)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedDoubleBag withAll(DoubleIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedDoubleBag withoutAll(DoubleIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(double item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(double item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().sizeDistinct();
        }
    }

    public int occurrencesOf(double item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(DoubleIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableDoubleBag select(DoublePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().select(predicate);
        }
    }

    @Override
    public MutableDoubleBag reject(DoublePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleBag().hashCode();
        }
    }

    @Override
    public LazyDoubleIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyDoubleIterableAdapter(this);
        }
    }

    @Override
    public MutableDoubleBag asUnmodifiable()
    {
        return new UnmodifiableDoubleBag(this);
    }

    @Override
    public MutableDoubleBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableDoubleBag toImmutable()
    {
        return DoubleBags.immutable.withAll(this);
    }
}
