package com.baijiayun.openapi.sdk;

import com.baijiayun.openapi.sdk.base.*;
import com.baijiayun.openapi.sdk.utils.BjyServiceUtil;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

/**
 * desc:
 *
 * @author yanmaoyuan
 */
@Slf4j
@Data
@Builder
public class BjyCallContext {
    private BjyPartnerInfo baseReqDto;
    private String                     path;
    private Request.Method method;
    private Map<String, Object>        params;
    private Map<String, Object>        headers;
    private Type returnType;

    public static BjyCallContext from(java.lang.reflect.Method method, Object[] args) {
        Request request = method.getAnnotation(Request.class);
        if (request == null) {
            throw new RuntimeException("method must be annotated with @Request");
        }

        return BjyCallContext
                .builder()
                .baseReqDto(baseReqDto(args))
                .method(request.method())
                .path(pathOf(request, method, args))
                .params(requestParamsOf(method, args))
                .headers(headerParamsOf(method, args))
                .returnType(method.getGenericReturnType())
                .build();
    }

    private static BjyPartnerInfo baseReqDto(Object[] args) {
        for (Object arg : args) {
            if (arg instanceof BjyPartnerInfo)
                return (BjyPartnerInfo) arg;
        }
        throw new RuntimeException("no BjyPartnerInfo found in args");
    }

    private static String pathOf(Request request, java.lang.reflect.Method method, Object[] args) {

        Map<String, String> pathParams = pathParamsOf(method, args);

        String path = request.path();

        for (Entry<String, String> it : pathParams.entrySet()) {
            String name = it.getKey();
            String value = it.getValue();
            path = path.replace("{" + name + "}", value);
        }

        return path;
    }

    private static Map<String, String> pathParamsOf(java.lang.reflect.Method method, Object[] args) {
        Map<String, String> params = new HashMap<>();

        Parameter[] parameters = method.getParameters();

        for (int i = 0; i < parameters.length; i++) {
            Parameter it    = parameters[i];
            PathParam param = it.getAnnotation(PathParam.class);

            if (param != null) {
                params.put(param.value(), String.valueOf(args[i]));
            }
        }

        return params;
    }

    private static Map<String, Object> requestParamsOf(java.lang.reflect.Method method, Object[] args) {
        Map<String, Object> params = new HashMap<>();

        Parameter[] parameters = method.getParameters();

        for (int i = 0; i < parameters.length; i++) {
            Parameter    it           = parameters[i];
            RequestParam requestParam = it.getAnnotation(RequestParam.class);
            Object       arg          = args[i];

            if (args[i] instanceof BjyPartnerInfo) {
                // 不要把 partnerInfo 作为参数传递
                continue;
            }

            if (requestParam != null && requestParam.value() != null) {
                params.put(requestParam.value(), arg);
            } else if (arg != null) {
                if (arg instanceof Number ||
                        arg instanceof Boolean ||
                        arg instanceof CharSequence ||
                        arg instanceof File ||
                        arg instanceof Date ||
                        arg instanceof Enum ||
                        arg instanceof Collection ||
                        arg.getClass().isArray()) {
                    params.put(it.getName(), arg);
                } else {
                    try {
                        Map<String, String> map = BjyServiceUtil.beanToMap(args[i]);
                        Map<String, File> files = BjyServiceUtil.beanToFileMap(args[i]);
                        if (!map.isEmpty()) {
                            params.putAll(map);
                        }
                        if (!files.isEmpty()) {
                            params.putAll(files);
                        }
                    } catch (IOException e) {
                        log.error("bean to map error", e);
                        throw new RuntimeException(e);
                    }
                }
            }
        }

        return params;
    }

    private static Map<String, Object> headerParamsOf(java.lang.reflect.Method method, Object[] args) {
        Map<String, Object> params = new HashMap<>();

        Parameter[] parameters = method.getParameters();

        for (int i = 0; i < parameters.length; i++) {
            Parameter   it          = parameters[i];
            HeaderParam headerParam = it.getAnnotation(HeaderParam.class);

            if (headerParam != null && headerParam.value() != null) {
                params.put(headerParam.value(), args[i]);
            }
        }

        return params;
    }
}