
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2016 All Rights Reserved.
 */
package com.baijia.tianxiao.common.service;

import java.io.OutputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.constants.DataProcType;
import com.google.common.collect.Lists;

import lombok.Data;

/**
 * @title ImportDataProcessService
 * @desc 针对导入的数据进行检测的接口
 * @author cxm
 * @date 2016年3月15日
 * @version 1.0
 */
public interface ImportDataProcessService {

    @Data
    public static class BatchSaveResult {
        private List<Integer> failIndexs = Lists.newArrayList();

        private List<Integer> repeatIndexs = Lists.newArrayList();
    }

    /**
     * 单独一行数据保存的结果
     * 
     * @title SingleSaveResult
     * @desc TODO
     * @author cxm
     * @date 2016年3月15日
     * @version 1.0
     */
    @Data
    public static class SingleSaveErrorResult {
    	
    	private Object[] lineData;

        private boolean isRepeat;

        private String errorMsg;

        private Exception exception;

        private boolean isSuccess;

        private boolean isPublicClue = false;//是否有公海线索

        
        public SingleSaveErrorResult(){
        
        }
        public SingleSaveErrorResult(Object[] lineData){
        	this.lineData = lineData;
        }
        /**
         * @param isRepeat
         * @param errorMsg
         */
        public SingleSaveErrorResult(boolean isRepeat, String errorMsg) {
            this.isRepeat = isRepeat;
            this.errorMsg = errorMsg;
        }

        public SingleSaveErrorResult(boolean isRepeat, Exception exception) {
            this.isRepeat = isRepeat;
            this.exception = exception;
        }

        public static SingleSaveErrorResult createSuccessResult(String msg) {
            SingleSaveErrorResult result = new SingleSaveErrorResult(false, msg);
            result.setSuccess(true);
            return result;
        }

        public static SingleSaveErrorResult createSuccessResult(String msg,boolean isPublicClue) {
            SingleSaveErrorResult result = new SingleSaveErrorResult(false, msg);
            result.setSuccess(true);
            result.setPublicClue(isPublicClue);
            return result;
        }
        
        public static SingleSaveErrorResult createFailResult(String msg) {
            SingleSaveErrorResult result = new SingleSaveErrorResult(false, msg);
            result.setSuccess(false);
            return result;
        }

    }
    

    /**
     * 返回实现类中支持处理的数据类型具体的类型请参考 DataProcType
     * 
     * @return
     */
    public DataProcType getProcessType();

    /**
     * 将导入文件的首行进行校验,看是否符合数据格式规范
     * 
     * @param headers
     * @return
     */
    public boolean validateHeader(List<String> headers);
    
    /**
     * 将导入文件的首行进行校验,看是否符合数据格式规范
     * 
     * @param headers
     * @param orgId
     * @return
     */
    public boolean validateHeader(List<String> headers,Long orgId);

    /**
     * 保存单行数据. 如果保存失败,抛出异常.
     * 
     * @param lineData
     * @param updateRepeat
     * @return null表示保存成功
     */
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    public SingleSaveErrorResult saveSingleData(Long orgId, Long cascadeId, List<String> headers, Object[] lineData,
        boolean updateRepeat);

    /**
     * 批量保存数据,后期在数据量大的时候考虑这种方案
     * 
     * @param datas
     * @param updateRepeat
     * @return
     */
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    public BatchSaveResult batchImportDatas(Long orgId, Long cascadeId, List<String> headers, List<Object[]> datas,
        boolean updateRepeat);

    public void afterComplete();
    
    
    
    public int[] getUniqueIndex();
    
    public SingleSaveErrorResult validate(Long orgId, Long cascadeId, List<String> headers, SingleSaveErrorResult result, List<SingleSaveErrorResult> dataMap, boolean updateRepeat);
    
    public void downloadValidateResult(OutputStream os, Long orgId, String taskId, Collection<SingleSaveErrorResult> dataMap);
    
    public void downloadImportResult(OutputStream os, Long orgId, String taskId, Collection<SingleSaveErrorResult> data);
    
    void downloadImportTemplate(OutputStream os);
    
    public String validateResult(List<SingleSaveErrorResult> list);
}
