package com.baijia.tianxiao.redis.lock;

import com.google.gson.Gson;

import java.util.UUID;

import lombok.Data;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author weihongyan
 * @apiNote <(▰˘◡˘▰)> redis分布式锁接口
 * @since 22/05/2017 7:59 PM
 * @see RedisDistributeLockImpl
 */
public interface RedisDistributeLock {

    /**
     * 获取锁
     * 
     * @param key key
     * @return 获取到的锁, 获取不到时立刻返回null;
     */
    LockObject lockFastFail(String key);

    /**
     * 获取锁
     * 
     * @param key key
     * @param holdLockExpMills 如果获取到了锁, 预计最大占有时间.
     * @return 获取到的锁, 获取不到时立刻返回null;
     */
    LockObject lockFastFail(String key, long holdLockExpMills);

    /**
     * 获取锁
     * 
     * @param key key
     * @param waitMills 获取不到锁时自旋等待时间. 小于0就一直等.
     * @return 获取到的锁, 获取不到时返回null;
     */
    LockObject lockOrWait(String key, long waitMills);

    /**
     * 获取锁
     * 
     * @param key key
     * @param waitMills 获取不到锁时自旋等待时间. 小于0就一直等.
     * @param holdLockExpMills 如果获取到了锁, 预计最大占有时间.
     * @return 获取到的锁, 获取不到时返回null;
     */
    LockObject lockOrWait(String key, long waitMills, long holdLockExpMills);

    /**
     * 获取锁
     * 
     * @param key key
     * @param waitMills 获取不到锁时自旋等待时间. 小于0就一直等.
     * @param holdLockExpMills 如果获取到了锁, 预计最大占有时间.
     * @param spinMills 自旋周期sleep时间.
     * @return 获取到的锁, 获取不到时返回null;
     */
    LockObject lockOrWait(String key, long waitMills, long holdLockExpMills, long spinMills);

    /**
     * 释放锁
     * 
     * @param key key
     * @param lockObject 上文获取到的锁
     * @return 当上文取到的锁,持有超时后(会被其他锁竞争者抢走),再次释放会释放失败返回false.
     */
    boolean unlock(String key, LockObject lockObject);

    /**
     * 获取到的锁, 释放时有用
     */
    @Data
    @Slf4j
    class LockObject {
        public LockObject(@NonNull Long lockExpMills) {
            this.lockExpMills = lockExpMills;
        }

        String uuid = UUID.randomUUID().toString();
        Long ct = System.currentTimeMillis();
        Long lockExpMills;

        public String toJson() {
            return new Gson().toJson(this);
        }

        public static LockObject formJson(String json) {
            try {
                return new Gson().fromJson(json, LockObject.class);
            } catch (IllegalStateException e) {
                log.error("deGson failed! maybe this lock key already has value:{} this can not be deGsoned to"
                    + " LockObject. try to change this lock key or delete this key e:{}", json, e);
                throw e;
            }
        }
    }
}
