/**
 * Baijiahulian.com Inc. Copyright (c) 2015-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.base;

import com.baijia.tianxiao.dto.WebResponse;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.SalWechatErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.CustomException;
import com.baijia.tianxiao.exception.ParameterException;
import com.baijia.tianxiao.exception.PermissionException;
import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.WebResponseHelper;
import com.baijia.tianxiao.util.date.TimeStampPropertyEditor;
import com.baijia.tianxiao.util.encrypt.EncryptUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.baijia.tianxiao.util.response.ResponseUtil;

import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;

import java.io.IOException;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;

/**
 * @title : BaseController
 * @description : ajax请求 异常捕获
 * @author : zhenyujian
 * @date : 2015年12月9日 下午3:37:40
 */
@Slf4j
public class BaseController {

    // 日志格式 分隔符
    private static final String LOG_SEPARATOR = " - ";

    // 日志级别
    private enum LogLevel {
        TRACE, DEBUG, INFO, WARN, ERROR
    }

    // 自定义异常列表
    private static final Class<?>[] CUSTOM_EXCEPTION = { BussinessException.class, ParameterException.class,
        PermissionException.class, WebServiceException.class, WechatException.class };

    /**
     * @Description : 异常捕获 只处理ajax请求,返回json。 (其他请求如访问页面需在方法中完成try catch)
     * @Author : zhenyujian
     * @CreationDate : 2015年12月9日 下午10:44:48
     *
     * @Return : void
     */
    @ExceptionHandler({ Exception.class })
    public void exception(HttpServletRequest request, HttpServletResponse response, Exception exception)
        throws IOException, NoSuchMethodException, SecurityException, ClassNotFoundException {

        // request信息
        String uri = request.getRequestURI();
        String requestParams = getRequestParamStr(request);

        // response 返回结果
        WebResponse<?> webResponse = null;

        // 日志 - 参数 日志级别
        StringBuilder logStrBulider = new StringBuilder();
        Object[] logParams = null;
        LogLevel logLevel;

        // 临时变量
        String exceptionName = null;
        Map<String, Object> logParamsMap = new LinkedHashMap<String, Object>();
        String simpleName = exception.getClass().getSimpleName();
        // 判断异常类型
        if (exception instanceof CustomException) {
            CustomException baseException = (CustomException) exception;
            exceptionName = CustomException.class.getSimpleName();
            webResponse = WebResponseHelper.error(baseException.getErrorCode(), baseException.getMessage());

            for (Class<?> customException : CUSTOM_EXCEPTION) {
                if (exception.getClass() == customException) {
                    exceptionName = customException.getSimpleName();
                }
            }

            logParamsMap.put("requestParams", requestParams);
            logParamsMap.put("errorCode", baseException.getErrorCode());
            logParamsMap.put("msg", baseException.getMessage());
            logLevel = LogLevel.WARN;
        } else if (exception instanceof IllegalArgumentException) {
            webResponse = WebResponseHelper.error(CommonErrorCode.PARAM_ERROR, exception.getMessage());
            exceptionName = exception.getClass().getSimpleName();
            logParamsMap.put("requestParams", requestParams);
            logLevel = LogLevel.WARN;
        } else if (exception instanceof MissingServletRequestParameterException) {// 处理spring
            webResponse = WebResponseHelper.error(CommonErrorCode.PARAM_ERROR, exception.getMessage());
            exceptionName = MissingServletRequestParameterException.class.getSimpleName();
            logParamsMap.put("requestParams", requestParams);
            logLevel = LogLevel.WARN;
        } else if (simpleName.equals("ClientAbortException")) {
            webResponse = WebResponseHelper.error(CommonErrorCode.UNKNOW);
            logParamsMap.put("requestParams", requestParams);
            logLevel = LogLevel.WARN;
        } else {
            webResponse = WebResponseHelper.error(CommonErrorCode.SYSTEM_ERROR);
            exceptionName = Exception.class.getSimpleName();
            logParamsMap.put("requestParams", requestParams);
            logLevel = LogLevel.ERROR;
        }

        // 日志 error 级别过滤
        if (exception instanceof WechatException) {
            WechatException we = (WechatException) exception;
            if (we.getErrorCode().getSubsystemErrorCode() == SalWechatErrorCode.WECHAT_REFRESH_TOKEN_IS_INVALID
                .getSubsystemErrorCode()) {
                logLevel = LogLevel.WARN;
            }
        }

        // 拼装日志 内容串 变长参数数组
        // [exceptionName] - [uri] - [params]
        // Exception - /wechat/signinCardInfo.do - requestParams:{lessonId:[5623], classId:[920291], studentId:[940106],
        // action:[signinConfirm], _:[1446695817727], openId:[owQJmxLVFoDk8LONwXRmseA7yP14]}
        logStrBulider.append(exceptionName).append(LOG_SEPARATOR);
        logStrBulider.append(uri).append(LOG_SEPARATOR);

        if (logParamsMap != null && !logParamsMap.isEmpty()) {
            logParams = new Object[logParamsMap.size()];
            int i = 0;
            for (String key : logParamsMap.keySet()) {
                if (i != 0) {
                    logStrBulider.append(",");
                }

                logStrBulider.append(key).append(":{}");
                logParams[i] = logParamsMap.get(key);
                i++;
            }
        }

        // 打印日志
        switch (logLevel) {
            case TRACE:
                log.trace(logStrBulider.toString(), logParams);
                break;
            case DEBUG:
                log.debug(logStrBulider.toString(), logParams);
                break;
            case INFO:
                log.info(logStrBulider.toString(), logParams);
                break;
            case WARN:
                log.warn(logStrBulider.toString(), logParams);
                break;
            case ERROR:
                log.error(logStrBulider.toString(), logParams);
                log.error(exception.getMessage(), exception);
                // exception.printStackTrace();
                break;
            default:
                exception.printStackTrace();
                break;
        }

        // 返回json给客户端
        try {
            response.setContentType("application/json;charset=UTF-8");
            JacksonUtil.writeObj(response.getOutputStream(), webResponse);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * @Description : 拼装 request参数串 { "key1":[value], "key2":[value1,value2] }
     * 
     * @Author : zhenyujian
     * @CreationDate : 2015年12月9日 下午3:39:22
     *
     * @Return : String
     */
    private String getRequestParamStr(HttpServletRequest httpRequest) {
        StringBuilder builder = new StringBuilder();
        builder.append("{");

        Map<String, String[]> paramMap = httpRequest.getParameterMap();
        boolean flag = false;// 参数前是否添加逗号
        String[] values = null;

        if (paramMap != null) {
            for (String key : paramMap.keySet()) {
                if (flag) {
                    builder.append(", ");
                } else {
                    flag = true;
                }
                values = paramMap.get(key);
                builder.append(key).append(":");
                builder.append("[");
                if (values != null && values.length > 0) {
                    for (int j = 0; j < values.length; j++) {
                        builder.append(values[j]);
                        if (j != values.length - 1) {
                            builder.append(",");
                        }
                    }
                }
                builder.append("]");
            }
        }

        builder.append("}");
        return builder.toString();
    }

    /**
     * @Description : 解析pageDto json字符串
     * @Author : zhenyujian
     * @Date : 2015年12月13日 下午5:38:14
     *
     * @Return : PageDto
     */
    PageDto parsePageDto(String pageDtoJsonStr) {
        PageDto dto = new PageDto();
        if (StringUtils.isNotBlank(pageDtoJsonStr)) {
            JSONObject pageJson = JSONObject.fromObject(pageDtoJsonStr);
            dto.setPageNum(pageJson.optInt("pageNum", -1));
            dto.setPageSize(pageJson.optInt("pageSize", -1));
        }
        return dto;
    }

    /**
     * 打印结果
     * 
     * @param req
     * @param resp
     * @param data
     */
    void printJson(HttpServletRequest req, HttpServletResponse resp, Object data) {
        resp.setContentType("application/json;charset=utf-8");
        ResponseUtil.write(resp, data);
    }

    /**
     * token
     * 
     * @param id
     * @return
     * @throws Exception
     */
    public static String encodeToken(Long id) throws Exception {
        String token = EncryptUtils.encodeWithAES(id.toString());
        return token;
    }

    @InitBinder
    protected void initBinder(WebDataBinder binder) {
        binder.registerCustomEditor(Date.class, new TimeStampPropertyEditor());
    }

    // private String newCookie(HttpServletResponse response) {
    // String cookieStr = UUID.randomUUID().toString();
    // cookieStr = cookieStr.replaceAll("-", "").toLowerCase();
    // Cookie cookie = new Cookie(BizConf.BAIJIA_YUNYING_ID, cookieStr);
    // cookie.setPath(BizConf.BAIJIA_YUNYING_ID_PATH);
    // cookie.setDomain(BizConf.BAIJIA_YUNYING_DOMAIN);
    // response.addCookie(cookie);
    // return cookieStr;
    // }

    private static final String TIANXIAO_APP_FLAG = "tianxiao";

    /**
     * 判断请求的客户端是否为天校app
     *
     * @param request
     * @return
     */
    public boolean isApp(HttpServletRequest request) {
        String userAgent = StringUtils.defaultIfBlank(request.getHeader("User-Agent"), "").toLowerCase();
        if (StringUtils.isEmpty(userAgent)) {
            return false;
        } else {
            return userAgent.contains(TIANXIAO_APP_FLAG);
        }
    }
}
