package com.baijia.tianxiao.excel;

import com.baijia.tianxiao.util.GenericsUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.POIXMLException;
import org.apache.poi.hssf.usermodel.HSSFClientAnchor;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Created by liuxp on 17/1/3.
 */
@Slf4j
public class ExcelExporter {

    protected <A> void exportToExcel(OutputStream outputStream, String fileName, String sheetName, List<A> datas,
                                     Excelable<A> exporter) {
        Workbook wb = new HSSFWorkbook();
        if (StringUtils.isBlank(sheetName)) {
            sheetName = "ExportDataList";
        }
        createSheet(wb, sheetName, datas, exporter);
        if (StringUtils.isBlank(fileName)) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmss");
            fileName = sdf.format(new Date()) + ".xls";
        }
        try {
            wb.write(outputStream);
        } catch (IOException e) {
            String simplename = e.getClass().getSimpleName();
            if ("ClientAbortException".equals(simplename)) {
                log.warn("Error while exporting data.And it's a ClientAbortException");
            }
        }
    }

    protected <A> void exportToExcel(HttpServletResponse response, String fileName, String sheetName, List<A> datas,
                                     Excelable<A> exporter) {
        if (GenericsUtils.notNullAndEmpty(datas) && datas.size() > 6000) {
            this.exportToExcelZip(response, fileName, sheetName, datas, exporter);
            return;
        }
        Workbook wb = new HSSFWorkbook();
        if (StringUtils.isBlank(sheetName)) {
            sheetName = "ExportDataList";
        }
        createSheet(wb, sheetName, datas, exporter);
        if (StringUtils.isBlank(fileName)) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmss");
            fileName = sdf.format(new Date()) + ".xls";
        }
        response.setContentType("application/vnd.ms-excel");
        response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
        try {
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            String simplename = e.getClass().getSimpleName();
            if ("ClientAbortException".equals(simplename)) {
                log.warn("Error while exporting data.And it's a ClientAbortException");
            }
        }
    }

    protected <A> void exportToExcelZip(HttpServletResponse response, String fileName, String sheetName, List<A> datas,
                                        Excelable<A> exporter) {
        if (StringUtils.isBlank(sheetName)) {
            sheetName = "ExportDataList";
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd_HHmmss_");
        if (StringUtils.isBlank(fileName)) {
            fileName = sdf.format(new Date());
        }
        List<File> files = new ArrayList<File>();
        FileOutputStream fileOut = null;
        File dic = new File("../exportExcel");
        if (!dic.exists()) {
            dic.mkdir();
        }
        try {
            int sheetNum = datas.size() / 60000;
            for (int i = 0; i < sheetNum + 1; i++) {
                Workbook wb = new HSSFWorkbook();
                File file = new File("../exportExcel/" + fileName + "_" + i + ".xls");
                file.createNewFile();
                files.add(file);
                fileOut = new FileOutputStream(file);
                if (i == sheetNum) {
                    createSheet(wb, sheetName, datas.subList(60000 * sheetNum, datas.size()), exporter);
                } else {
                    createSheet(wb, sheetName, datas.subList(60000 * i, 60000 * (i + 1)), exporter);
                }
                wb.write(fileOut);
                try {
                    fileOut.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } catch (IOException e) {
            String simplename = e.getClass().getSimpleName();
            if ("ClientAbortException".equals(simplename)) {
                log.warn("Error while exporting data.And it's a ClientAbortException");
            }
        }
        byte[] buf = new byte[1024];
        File zipFile = new File("../exportExcel/" + fileName + "_Excels.zip");
        try {
            if (!zipFile.exists()) {
                zipFile.createNewFile();
            }
            ZipOutputStream out = new ZipOutputStream(new FileOutputStream(zipFile));
            for (File file : files) {
                FileInputStream in = new FileInputStream(file);
                out.putNextEntry(new ZipEntry(file.getName()));
                int len;
                while ((len = in.read(buf)) > 0) {
                    out.write(buf, 0, len);
                }
                out.closeEntry();
                in.close();
                file.delete();
            }
            out.close();
        } catch (Exception e) {
            String simplename = e.getClass().getSimpleName();
            if ("ClientAbortException".equals(simplename)) {
                log.warn("Error while creating zipFile.And it's a ClientAbortException");
            }
        }
        try {
            InputStream fis = new BufferedInputStream(new FileInputStream(zipFile));
            byte[] buffer = new byte[fis.available()];
            fis.read(buffer);
            fis.close();
            response.reset();
            response.setContentType("application/octet-stream");
            response.setHeader("Content-Disposition", "attachment; filename=" + zipFile.getName());
            OutputStream output = new BufferedOutputStream(response.getOutputStream());
            output.write(buffer);
            output.flush();
            output.close();
            zipFile.delete();
        } catch (IOException e) {
            String simplename = e.getClass().getSimpleName();
            if ("ClientAbortException".equals(simplename)) {
                log.warn("Error while transporting zipFile.And it's a ClientAbortException");
            }

        }
    }

    protected <A> Sheet createSheet(Workbook wb, String sheetName, List<A> datas, Excelable<A> exporter) {
        Sheet sheet;
        if (sheetName != null) {
            sheet = wb.createSheet(sheetName);
        } else {
            sheet = wb.createSheet();
        }
        for (int i = 0; i < exporter.exportRowName().length; i++) {
            sheet.setColumnWidth(i, 20 * 256);
        }
        if (datas != null && datas.isEmpty() == false) {
            List<ExcelCell[]> result = preprocessData(datas, exporter);
            int size = result.size();
            Map<ExcelCellStyle, CellStyle> styleMap = new HashMap<ExcelCellStyle, CellStyle>();
            for (int i = 0; i < size; i++) {
                createRow(wb, sheet, i, result.get(i), styleMap);
            }
        }
        return sheet;
    }

    protected <A> List<ExcelCell[]> preprocessData(List<A> datas, Excelable<A> exporter) {
        List<ExcelCell[]> result = new ArrayList<ExcelCell[]>(datas.size() + 1);
        result.add(exporter.exportRowName());
        for (A data : datas) {
            result.add(exporter.exportRowValue(data));
        }
        return result;
    }

    protected Row createRow(Workbook wb, Sheet sheet, int rowNum, ExcelCell[] values,
                            Map<ExcelCellStyle, CellStyle> styleMap) {
        Row row = sheet.createRow(rowNum);
        ExcelCell value = null;
        int size = values.length;
        Drawing p = sheet.createDrawingPatriarch();
        for (int i = 0; i < size; i++) {
            Cell cell = row.createCell(i);
            value = values[i];
            if (value == null) {
                continue;
            }
            cell.setCellValue(value.getValue().toString());
            if (StringUtils.isBlank(value.getComment()) == false) {
                cell.setCellComment(createCellComment(p, value, (short) i, rowNum));
            }
            if (value.getCellType() != null) {
                cell.setCellType(value.getCellType());
            }
            ExcelCellStyle excelCellStyle = value.getCellStyle();
            if (excelCellStyle != null) {
                CellStyle cellStyle = styleMap.get(excelCellStyle);
                if (cellStyle == null) {
                    cellStyle = wb.createCellStyle();
                    styleMap.put(excelCellStyle, cellStyle);
                    Font font = null;
                    if (excelCellStyle.getAlignment() != null) {
                        cellStyle.setAlignment(excelCellStyle.getAlignment());
                    }
                    if (excelCellStyle.getVerticalAlignment() != null) {
                        cellStyle.setVerticalAlignment(excelCellStyle.getVerticalAlignment());
                    }
                    if (excelCellStyle.getFontColor() != null) {
                        if (font == null) {
                            font = wb.createFont();
                        }
                        font.setColor(excelCellStyle.getFontColor());
                    }
                    if (excelCellStyle.getBoldWeight() != null) {
                        if (font == null) {
                            font = wb.createFont();
                        }
                        font.setBoldweight(excelCellStyle.getBoldWeight());
                    }
                    if (font != null) {
                        cellStyle.setFont(font);
                    }
                }
                cell.setCellStyle(cellStyle);
            }
        }
        return row;
    }

    protected Comment createCellComment(Drawing draw, ExcelCell value, short col, int row) {
        short col2 = col;
        int row2 = row;
        col2++;
        row2++;
        int len = 0;
        try {
            len = value.getComment().getBytes("GBK").length;
        } catch (UnsupportedEncodingException e1) {
            e1.printStackTrace();
        }
        if (len > 24) {
            col2 = (short) (col + 3);
            row2 = row + (len - 1) / 24 + 1;
        } else {
            col2 = (short) (col + (len + 2) / 9 + 1);
        }
        Comment comment = draw.createCellComment(new HSSFClientAnchor(0, 0, 0, 127, col, row, col2, row2));
        comment.setString(new HSSFRichTextString(value.getComment()));
        comment.setVisible(value.isCommentVisible());
        return comment;
    }

    public static void main(String[] args) {
        Workbook wb = new HSSFWorkbook();
        Sheet sheet = wb.createSheet("ExportDataList");

        // Create a row and put some cells in it. Rows are 0 based.
        Row headerTitle = sheet.createRow(0);
        Cell cell = headerTitle.createCell(0);
        int dx1 = 0;
        int dy1 = 0;
        int dx2 = 0;
        int dy2 = 127;
        short col = 0;
        int row = 0;
        short col2 = 1;
        int row2 = 1;
        String value = "123456789012345";
        int len = 0;
        try {
            len = value.getBytes("GBK").length;
        } catch (UnsupportedEncodingException e1) {
            e1.printStackTrace();
        }
        System.out.println(len);
        if (len > 24) {
            col2 = (short) (col + 3);
            row2 = row + (len - 1) / 24 + 1;
        } else {
            col2 = (short) (col + (len + 2) / 9 + 1);
            dx2 = ((len + 3) % 9) * 113;
        }
        cell.setCellValue(value);
        Drawing draw = sheet.createDrawingPatriarch();

        Comment comment = draw.createCellComment(new HSSFClientAnchor(dx1, dy1, dx2, dy2, col, row, col2, row2));
        comment.setString(new HSSFRichTextString(value));
        comment.setVisible(false);
        cell.setCellComment(comment);
        OutputStream os = null;
        try {
            os = new FileOutputStream("d:\\test" + System.currentTimeMillis() + ".xls");
            wb.write(os);
            os.flush();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
                os = null;
            }
            if (wb != null) {
                try {
                    wb.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
                wb = null;
            }
        }

    }

    protected static Workbook getExcelWorkbook(String name, InputStream input)
            throws IOException, OpenXML4JException, POIXMLException {
        if (name.endsWith(".xls")) {
            return new HSSFWorkbook(input);
        } else if (name.endsWith(".xlsx")) {
            return new XSSFWorkbook(input);
        }
        return null;
    }

    protected static <T> void procExcelWorkbook(Workbook workbook, CellProcess<T> process, T result) {
        Sheet sheet = null;
        Row row = null;
        Cell cell = null;
        int sheetSize = workbook.getNumberOfSheets();
        int rowSize = 0;
        int cellSize = 0;
        // 循环工作表Sheet
        for (int numSheet = 0; numSheet < sheetSize; numSheet++) {
            sheet = workbook.getSheetAt(numSheet);
            if (sheet == null) {
                continue;
            }
            // 循环行Row
            rowSize = sheet.getLastRowNum();
            for (int rowNum = sheet.getFirstRowNum(); rowNum <= rowSize; rowNum++) {
                row = sheet.getRow(rowNum);
                if (row == null) {
                    continue;
                }
                cellSize = row.getLastCellNum();
                for (int cellNum = row.getFirstCellNum(); cellNum <= cellSize; cellNum++) {
                    cell = row.getCell(cellNum);
                    if (cell == null) {
                        continue;
                    }
                    process.proc(cell, result);
                }
            }
        }
    }

    protected static String getExcelCellValue(Cell cell) {
        switch (cell.getCellType()) {
            case Cell.CELL_TYPE_BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());
            case Cell.CELL_TYPE_NUMERIC:
                return new BigDecimal(cell.getNumericCellValue()).toString();
            default:
                return String.valueOf(cell.getStringCellValue());
        }
    }

    public static interface CellProcess<T> {
        public void proc(Cell cell, T result);
    }
}
