package com.baijia.tianxiao.util.rest;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.constants.UserRoleEnum;
import com.baijia.tianxiao.dto.RestfulResult;
import com.baijia.tianxiao.dto.signup.CreatePurchaseResponseDto;
import com.baijia.tianxiao.dto.signup.CreateTradePurchaseResponseDto;
import com.baijia.tianxiao.dto.signup.PayResponseDto;
import com.baijia.tianxiao.dto.signup.PayResultDto;
import com.baijia.tianxiao.dto.signup.PayResultMapResponseDto;
import com.baijia.tianxiao.dto.signup.PurchaseType;
import com.baijia.tianxiao.dto.signup.QueryTradeNoDataDto;
import com.baijia.tianxiao.dto.signup.QueryTradeNoMapResponseDto;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.enums.SignupErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.util.collection.CollectorUtil;
import com.baijia.tianxiao.util.http.GSXService;
import com.baijia.tianxiao.util.httpclient.HttpClientUtils;
import com.baijia.tianxiao.util.json.JacksonUtil;
import com.baijia.tianxiao.util.properties.UrlProperties;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import lombok.NonNull;

/**
 * Created by liuxp on 15/12/3.
 */
public class RestUtils {

    private static final Logger logger = LoggerFactory.getLogger(RestUtils.class);

    private static final String SPLIT = ":";
    private static final String PAY_WINXIN = "30";

    /**
     * 支付侧的终端类型定义
     * 
     * @title PayTerminal
     * @desc 支付侧的终端类型定义
     * @author shizuwei
     * @date 2015年12月9日
     * @version 1.0
     */
    public enum PayTerminal {
        PC(1), ANDROID(2), IPHONE(3), IPAD(4);
        private int code;

        private PayTerminal(int code) {
            this.code = code;
        }

        public int getCode() {
            return this.code;
        }

        @Override
        public String toString() {
            return String.valueOf(this.code);
        }
    }

    public enum RestMethod {
        GET, POST
    }

    private static final String URL_SPLIT = "/";
    private static final String SIGN_SPLIT = "-";
    private static final String DEF_CHARSET = "UTF-8";
    private static final String HMAC_SHA1 = "HmacSHA1";

    private static final byte[] LOGIN_LOCK = new byte[0];
    private static final long EXPIRE_TIME = 1000 * 60 * 60 * 24 * 7;

    private String appId;
    private String appKey;
    private String baseUrl;

    public static volatile String authToken = null;
    public static volatile long loginTime = -1;

    public RestUtils(String appId, String appKey, String baseUrl) {
        this.appId = appId;
        this.appKey = appKey;
        this.baseUrl = baseUrl;
    }

    public <T> RestfulResult<T> rest(RestMethod restMethod, String controller, String action, List<String> urlVarList,
        Map<String, String> params, T dataType) throws Exception {
        // 创建url
        StringBuilder loginUrl = new StringBuilder();
        loginUrl.append(baseUrl);
        loginUrl.append(controller);
        loginUrl.append(URL_SPLIT).append(action);

        // 生成签名
        String signData = createSignData(restMethod.toString(), controller, action, urlVarList, params);
        logger.info("signData=" + signData);
        String sign = getSignature(signData.getBytes(DEF_CHARSET), appKey.getBytes(DEF_CHARSET));

        // 设置签名
        params.put("sign", sign);

        // 发送请求
        String resultJson = null;
        switch (restMethod) {
            case POST:
                resultJson = HttpClientUtils.doPost(loginUrl.toString(), params, DEF_CHARSET);
                logger.info("login url=" + loginUrl);
                logger.info("param=" + params);
                break;
            case GET:
                resultJson = HttpClientUtils.doGet(loginUrl.toString(), params, DEF_CHARSET);
                break;
            default:
                throw new Exception("不支持的REST方法");
        }

        // 解析结果
        ObjectMapper mapper = new ObjectMapper();
        logger.trace(resultJson);
        @SuppressWarnings("unchecked")
        RestfulResult<T> result = mapper.readValue(resultJson, RestfulResult.class);

        return result;
    }

    public String getAuthToken() throws Exception {
        // TODO 是否可以避免JAVA中双重检查成例的构造和赋值无序的问题？
        // if(loginTime - System.currentTimeMillis() < EXPIRE_TIME && authToken != null){
        // return authToken;
        // }
        synchronized (LOGIN_LOCK) {
            if (loginTime - System.currentTimeMillis() < EXPIRE_TIME && authToken != null) {
                return authToken;
            }
            String result = null;
            try {
                String controller = "auth";
                String action = "login";
                // 设置参数
                Map<String, String> params = new HashMap<String, String>();
                params.put("app_id", appId);
                params.put("app_key", appKey);
                params.put("timestamp", String.valueOf(System.currentTimeMillis()));
                Map<String, Object> dataType = null;// new HashMap<String, Object>(0);
                RestfulResult<Map<String, Object>> loginResult =
                    rest(RestMethod.POST, controller, action, null, params, dataType);
                if (loginResult == null) {
                    return null;
                }
                if (loginResult.getCode() != 0) {
                    throw new Exception(loginResult.getMsg());
                }
                result = loginResult.getData().get("auth_token").toString();
                return result;
            } finally {
                loginTime = System.currentTimeMillis();
                authToken = result;
            }
        }
    }

    public static String createSignData(String restMethod, String controllerName, String actionName,
        List<String> urlVarList, Map<String, String> params) {
        StringBuilder signData = new StringBuilder();
        signData.append(restMethod.toLowerCase());
        signData.append(SIGN_SPLIT).append(controllerName);
        signData.append(SIGN_SPLIT).append(actionName);

        if (urlVarList != null && urlVarList.isEmpty() == false) {
            for (String urlVar : urlVarList) {
                if (urlVar == null) {
                    signData.append(SIGN_SPLIT).append("");
                } else {
                    signData.append(SIGN_SPLIT).append(urlVar);
                }

            }
        }

        if (params != null && params.isEmpty() == false) {
            String[] paraNames = new String[params.size()];
            Arrays.sort(params.keySet().toArray(paraNames));
            for (String paraName : paraNames) {
                String value = params.get(paraName);
                if (value == null) {
                    value = "";
                }
                signData.append(SIGN_SPLIT).append(value);
            }
        }
        return signData.toString();
    }

    /**
     * 生成签名数据
     *
     * @param data 待加密的数据
     * @param key 加密使用的key
     * @return 生成BASE64编码的字符串
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     */
    public static String getSignature(byte[] data, byte[] key) throws InvalidKeyException, NoSuchAlgorithmException {
        SecretKeySpec signingKey = new SecretKeySpec(key, HMAC_SHA1);
        Mac mac = Mac.getInstance(HMAC_SHA1);
        mac.init(signingKey);
        byte[] rawHmac = mac.doFinal(data);
        return Base64.encodeBase64String(rawHmac);
    }

    /**
     * 生成随机密码字符串
     * 
     * @param len 字符串长度
     * @return 随机密码字符串
     */
    public static String randomPwd(int len) {
        SecureRandom random = new SecureRandom();
        byte[] salt = new byte[len];
        random.nextBytes(salt);
        char[] saltChars = new char[len];
        int letterPoolSize = TianXiaoConstant.LETTER_POOL.length;
        int numberPoolSize = TianXiaoConstant.NUM_POOL.length;
        for (int i = 0; i < len; i++) {
            if (i == 0) {
                saltChars[i] = TianXiaoConstant.LETTER_POOL[(salt[i] + 128) % letterPoolSize];
            } else {
                saltChars[i] = TianXiaoConstant.NUM_POOL[(salt[i] + 128) % numberPoolSize];
            }
        }
        return new String(saltChars);
    }

    public static RestfulResult<Map<String, Object>> createStudent(String mobile, String password, String email,
        String name, String invite_code) throws Exception {
        String controller = "student";
        String action = "create";
        if (StringUtils.isBlank(password)) {
            password = randomPwd(6);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("mobile:" + mobile + ",password:" + password + ",email:" + email + ",name:" + name
                + ",invite_code:" + invite_code);
        }
        RestfulResult<Map<String, Object>> createResult = null;
        if (StringUtils.isBlank(mobile) || StringUtils.isBlank(password)) {
            createResult = new RestfulResult<Map<String, Object>>();
            createResult.setCode(1);
            createResult.setMsg("参数不完整");
            return createResult;
        }
        if (email == null) {
            email = "";
        }
        if (name == null) {
            name = "";
        }
        if (invite_code == null) {
            invite_code = "";
        }

        // 设置参数
        SortedMap<String, String> params = new TreeMap<>();
        params.put("mobile", mobile);
        params.put("password", password);
        params.put("email", email);
        params.put("realname", name);
        params.put("invite_code", invite_code);
        createResult = GSXService.doService(controller, action, params);
        logger.info("create student return:{}", createResult);
        if (createResult.getCode() == 100050) {
            addRole(mobile, null, UserRoleEnum.STUDENT.getCode(), invite_code);
        }

        return createResult;
    }

    public static RestfulResult<Map<String, Object>> addRole(String mobile, Long orgId, Integer target_role,
        String invite_code) throws Exception {
        logger.trace("mobile:" + mobile + ",target_role" + target_role);
        RestfulResult<Map<String, Object>> addResult = null;
        if (StringUtils.isBlank(mobile) || (target_role != 0 && target_role != 2)) {
            addResult = new RestfulResult<Map<String, Object>>();
            addResult.setCode(1);
            addResult.setMsg("参数不完整");
            return addResult;
        }
        String controller = "user";
        String action = "addRole";

        SortedMap<String, String> params = new TreeMap<>();
        params.put("mobile", mobile);
        params.put("target_role", target_role.toString());
        params.put("invite_code", invite_code);
        if (orgId != null) {
            params.put("org_id", orgId.toString());
        }
        addResult = GSXService.doService(controller, action, params);
        logger.info("ResUtils.addRole   params:{}, addResult:{}", params, addResult);

        return addResult;
    }

    public static String getWeiXinPurchaseUrl(@NonNull Long purchaseId, @NonNull Long orgId,
        @NonNull Double totalPrice, Collection<Header> headers) throws BussinessException {
        Preconditions.checkArgument(totalPrice > 0, "price must great than 0");
        Preconditions.checkArgument(orgId > 0, "org id can not be null");
        String url = null;
        if (totalPrice < 0) {
            logger.warn("刷卡金额错误：totalPrice = {}", totalPrice);
            return null;
        }

        Map<String, String> params = Maps.newHashMap();
        params.put("purchase_id", purchaseId.toString());
        params.put("user_id", orgId.toString());
        params.put("pay_type", PAY_WINXIN + SPLIT + totalPrice);

        String weixinJson =
            HttpClientUtils.doPost(UrlProperties.getProperty("pay.getWeixinUrl.url"), params, HttpClientUtils.CHARSET,
                headers);
        if (StringUtils.isNotBlank(weixinJson)) {
            try {
                CreatePurchaseResponseDto weixinRes = JacksonUtil.str2Obj(weixinJson, CreatePurchaseResponseDto.class);
                if (weixinRes.getCode() == TianXiaoConstant.SUCC_CODE) {
                    url = weixinRes.getData().getPayUrl();
                }
            } catch (Exception e) {
                logger.error("trans weixinPurchaseRes error!", e);
            }
        }
        return url;
    }

    public static String getWeiXinPurchaseUrl(@NonNull Long orgId, @NonNull Double totalPrice,
        Collection<Header> headers,CreatePurchaseResponseDto createPurchaseResponseDto) throws BussinessException {
        Preconditions.checkArgument(totalPrice > 0, "price must great than 0");
        Preconditions.checkArgument(orgId > 0, "org id can not be null");
        if (totalPrice < 0) {
            logger.warn("刷卡金额错误：totalPrice = {}", totalPrice);
            return null;
        }

        return getWeiXinPurchaseUrl(createPurchaseResponseDto.getData().getPurchaseId(), orgId, totalPrice, headers);
    }

    public static Long getPurchaseId(@NonNull Long orgId, @NonNull Double totalPrice, Collection<Header> headers)
        throws BussinessException {
        return createPurchaseResponseDto(orgId, totalPrice, headers).getData().getPurchaseId();
    }

    public static CreatePurchaseResponseDto createPurchaseResponseDto(@NonNull Long orgId, @NonNull Double totalPrice,
        Collection<Header> headers) throws BussinessException {
        Preconditions.checkArgument(orgId > 0, "org id can not be null");
        Preconditions.checkArgument(totalPrice > 0, "price must great than 0");
        String response = null;
        try {
            Map<String, String> params = new HashMap<String, String>();
            params.put("org_id", orgId.toString());
            params.put("total_prices", totalPrice.toString());

            final String url = UrlProperties.getProperty("pay.createpurchase.url");
            logger.info("create purchase by params:{},url:{}", params, url);
            response = HttpClientUtils.doPost(url, params, HttpClientUtils.CHARSET, headers);
            logger.info("create purchase-->response {}", response);
        } catch (Exception e) {
            logger.warn("create purchase get error:{}", e);
            throw new BussinessException(SignupErrorCode.SPLITPURCHASE_ERROR);
        }

        CreatePurchaseResponseDto dto = checkPayResponse(response, CreatePurchaseResponseDto.class);
        Preconditions.checkNotNull(dto.getData().getPurchaseId(), "can not get purchase id from response:" + response);
        return dto;
    }

    private static <T extends PayResponseDto> T checkPayResponse(String response, Class<T> type)
        throws BussinessException {
        T splitSignupPurchaseResponseDto = null;
        if (StringUtils.isNotEmpty(response)) {
            try {
                splitSignupPurchaseResponseDto = JacksonUtil.str2Obj(response, type);
            } catch (Exception e) {
                logger.error("parse result:{} to object get error:{}", response, e);
                throw new BussinessException(CommonErrorCode.SYSTEM_ERROR);
            }
            if (splitSignupPurchaseResponseDto != null) {
                if (splitSignupPurchaseResponseDto.getCode() != 0) {
                    String message = "";
                    try {
                        message = new String(splitSignupPurchaseResponseDto.getMsg().getBytes("UTF-8"), "UTF-8");
                    } catch (UnsupportedEncodingException e) {
                        throw new BussinessException(CommonErrorCode.SYSTEM_ERROR);
                    }
                    // 特殊处理,兼容测试环境修改状态测试
                    if (!message.startsWith("当前机构收费订单已分拆完成!")) {
                        throw new BussinessException(CommonErrorCode.SYSTEM_ERROR);
                    }
                }
            }
            return splitSignupPurchaseResponseDto;
        }
        throw new BussinessException(SignupErrorCode.SPLITPURCHASE_ERROR);
    }

    public static Long getTradeNo(@NonNull Long orgId, @NonNull Long purchaseId, @NonNull Double totalPrice,
        Collection<Header> headers) throws BussinessException {
        Preconditions.checkArgument(orgId > 0, "org id can not be null");
        Preconditions.checkArgument(totalPrice > 0, "price must great than 0");
        String response = null;
        try {
            Map<String, String> params = new HashMap<String, String>();
            params.put("pro_purchase_id", purchaseId.toString());
            params.put("pro_purchase_type", String.valueOf(PurchaseType.SIGNUP.getCode()));
            params.put("money", totalPrice.toString());
            params.put("user_id", orgId.toString());
            params.put("user_role", String.valueOf(UserRoleEnum.ORG.getCode()));
            params.put("tid", PayTerminal.PC.toString());

            final String url = UrlProperties.getProperty("pay.createTradePurchase.url");
            logger.info("create trade purchase by params:{},url:{}", params, url);
            response = HttpClientUtils.doPost(url, params, HttpClientUtils.CHARSET, headers);
            logger.info("create trade-->response {}", response);
            CreateTradePurchaseResponseDto responseDto =
                JacksonUtil.str2Obj(response, CreateTradePurchaseResponseDto.class);
            return responseDto.getData().getPlatformTradeNo();
        } catch (Exception e) {
            logger.warn("create trade get error:{}", e);
            throw new BussinessException(SignupErrorCode.GETTRADENO_ERROR);
        }
    }

    public static Map<Long, Long> getTradeNoMap(Collection<Long> purchaseIds) {
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(purchaseIds), "purchaseIds can not be null");
        String response = null;
        try {
            Map<String, String> params = new HashMap<String, String>();
            params.put("purchase_ids", StringUtils.join(purchaseIds, ','));

            final String url = UrlProperties.getProperty("pay.queryTradeNo.url");
            logger.info("get tradeNo map by params:{},url:{}", params, url);
            if (url == null) {
                return null;
            }
            response = HttpClientUtils.doPost(url, params, HttpClientUtils.CHARSET);
            logger.info("get tradeNo map-->response {}", response);
            QueryTradeNoMapResponseDto responseDto = JacksonUtil.str2Obj(response, QueryTradeNoMapResponseDto.class);
            List<QueryTradeNoDataDto> data = responseDto.getData();
            if (CollectionUtils.isNotEmpty(data)) {
                return CollectorUtil.collectMap(data, new Function<QueryTradeNoDataDto, Long>() {
                    @Override
                    public Long apply(QueryTradeNoDataDto input) {
                        return input.getProPurchaseId();
                    }

                }, new Function<QueryTradeNoDataDto, Long>() {
                    @Override
                    public Long apply(QueryTradeNoDataDto input) {
                        return input.getPlatformTradeNo();
                    }
                });
            }
        } catch (Exception e) {
            logger.warn("get tradeNo map get error:{}", e);
        }
        return Maps.newHashMap();
    }
    
    
    
    public static PayResultDto getSignupPayResultDto (@NonNull Long signupPurchaseId) throws BussinessException, JsonParseException, JsonMappingException, IOException {
        Preconditions.checkArgument(signupPurchaseId > 0, "org id can not be null");
        String response = null;
        try {
            Map<String, String> params = new HashMap<String, String>();
            params.put("purchase_id", signupPurchaseId.toString());

            final String url = UrlProperties.getProperty("pay.orgsignupdetail.url");
            logger.info("getSignuoPayResultDto by signupPurchaseId:{},url:{}", signupPurchaseId, url);
            response = HttpClientUtils.doPost(url, params, HttpClientUtils.CHARSET);
            logger.info("getSignuoPayResultDto-->response {}", response);
        } catch (Exception e) {
            logger.warn("getSignuoPayResultDto get error:{}", e);
            throw new BussinessException(SignupErrorCode.SPLITPURCHASE_ERROR);
        }

        PayResultMapResponseDto responseDto = JacksonUtil.str2Obj(response, PayResultMapResponseDto.class);
        PayResultDto dto = responseDto.getData().getPayResultDto();
        return dto;
     }

    public static void main(String[] args) {
        double d1=1;

        System.out.println((100d-90d)/100*2000*100);
    }

}