package com.baijia.tianxiao.sal.wx.impl;

import com.baijia.tianxiao.constant.PublishStatus;
import com.baijia.tianxiao.constants.CategoryType;
import com.baijia.tianxiao.constants.TianXiaoConstant;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.OrgInfoDao;
import com.baijia.tianxiao.dal.org.dao.OrgStorageDao;
import com.baijia.tianxiao.dal.org.dao.TXCascadeCredentialDao;
import com.baijia.tianxiao.dal.org.dao.WxCategoryDao;
import com.baijia.tianxiao.dal.org.dao.WxContactDao;
import com.baijia.tianxiao.dal.org.dao.WxNewsDao;
import com.baijia.tianxiao.dal.org.po.OrgInfo;
import com.baijia.tianxiao.dal.org.po.OrgStorage;
import com.baijia.tianxiao.dal.org.po.WxCategory;
import com.baijia.tianxiao.dal.org.po.WxContactInfo;
import com.baijia.tianxiao.dal.org.po.WxNews;
import com.baijia.tianxiao.dal.storage.dao.StorageDao;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.wx.api.WxNewsService;
import com.baijia.tianxiao.sal.wx.model.WxNewsBaseDto;
import com.baijia.tianxiao.sal.wx.model.WxNewsDto;
import com.baijia.tianxiao.sal.wx.model.WxNewsReqDto;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.storage.StorageUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by wangsixia on 2017/1/13.
 */
@Service
public class WxNewsServiceImpl implements WxNewsService{

    private static Integer RICH_MAX_LENGTH = 30000;
    private static Integer CONTACT_MAX_LENGTH = 10000;
    private static String DEFAULT_NULL_TIME = "0000-00-00 00:00:00";

    @Autowired
    private WxNewsDao wxNewsDao;
    @Autowired
    private WxCategoryDao wxCategoryDao;
    @Autowired
    private OrgInfoDao orgInfoDao;
    @Autowired
    private TXCascadeCredentialDao txCascadeCredentialDao;
    @Autowired
    private WxContactDao wxContactDao;
    @Autowired
    private OrgStorageDao orgStorageDao;


    @Override
    @Transactional(rollbackFor = { Exception.class })
    public Long save(Long orgId, WxNewsDto param, Integer cascadeId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(param != null, "参数错误");
        if (param.getRichContent().length() > RICH_MAX_LENGTH) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "新闻内容过长");
        }
        WxNews po = null;
        if (param.getId() != null && param.getId() > 0) {
            po = this.wxNewsDao.getById(orgId, param.getId(), null);
        } else {
            po = new WxNews();
        }
        param.toPo(po, cascadeId, orgId);
        this.wxNewsDao.saveOrUpdateWithDefaultVal(po);
        return po.getId();
    }

    @Override
    public WxNewsDto getDetail(Long orgId, Long id) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(id != null && id > 0);
        WxNews po = this.wxNewsDao.getById(orgId, id, null);
        if (po == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "新闻不存在或已被删除");
        }
        Map<Long, WxCategory> categoryMap = this.cacheNewsCategory(orgId);
        Map<Long, String> coverMap = new HashMap<>();
        if (po.getCoverId() != null && po.getCoverId() > 0) {
            coverMap = this.cacheCoverUrl(Sets.newHashSet(po.getCoverId()));
        }
        Map<Integer, String> operatorMap = this.cacheOperator(orgId, Sets.newHashSet(po.getOperatorId()));
        WxNewsDto dto = WxNewsDto.getFromPo(po, categoryMap, coverMap, operatorMap);
        return dto;
    }

    @Override
    @Transactional(rollbackFor = { Exception.class })
    public void delete(Long orgId, Long id) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(id != null && id > 0);
        WxNews po = this.wxNewsDao.getById(orgId, id, null, "id");
        if (po == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "新闻不存在或已被删除");
        }
        po.setIsDel(DeleteStatus.DELETED.getValue());
        po.setUpdateTime(new Date());
        this.wxNewsDao.updateWithDefaultVal(po, "isDel", "updateTime");
    }

    @Override
    public List<WxNewsBaseDto> getBaseList(Long orgId, WxNewsReqDto param, Boolean descByPublish, PageDto pageDto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId错误");
        List<WxNewsBaseDto> data = new ArrayList<>();
        List<WxNews> list = this.wxNewsDao.getListByParam(orgId, null, param.getCategoryId(), param.getKey(), param.getStatus(), param.getStart(), param.getEnd(), descByPublish, pageDto);
        if (CollectionUtils.isNotEmpty(list)) {
            Set<Long> coverIds = new HashSet<>();
            Set<Integer> operatorIds = new HashSet<>();
            for (WxNews po : list) {
                if (po.getCoverId() != null && po.getCoverId() > 0) {
                    coverIds.add(po.getCoverId());
                }
                if (po.getOperatorId() != null && po.getOperatorId() >= 0) {
                    operatorIds.add(po.getOperatorId());
                }
            }
            Map<Long, WxCategory> categoryMap = this.cacheNewsCategory(orgId);
            Map<Long, String> coverMap = null;
            if (CollectionUtils.isNotEmpty(coverIds)) {
                coverMap = this.cacheCoverUrl(coverIds);
            }
            Map<Integer, String> operatorMap = null;
            if (CollectionUtils.isNotEmpty(operatorIds)) {
                operatorMap = this.cacheOperator(orgId, operatorIds);
            }
            for (WxNews po : list) {
                WxNewsBaseDto dto = WxNewsDto.getFromPo(po, categoryMap, coverMap, operatorMap);
                data.add(dto);
            }
        }
        return data;
    }

    @Override
    @Transactional(rollbackFor = { Exception.class })
    public void switchStatus(Long orgId, Collection<Long> ids, Integer status) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(ids), "请选择新闻");
        Preconditions.checkArgument(StringUtils.isNotBlank(PublishStatus.getByCode(status)), "状态错误");

        List<WxNews> list = this.wxNewsDao.getListByParam(orgId, ids, null, null, null, null, null, null, null);
        if (CollectionUtils.isEmpty(list) || list.size() != ids.size()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "参数错误");
        }
        Map<String, Object> condition = new HashMap<>();
        condition.put("id", ids);
        condition.put("status", status);
        condition.put("updateTime", new Date());
        if (status.intValue() == PublishStatus.PUBLISH.getCode()) {
            condition.put("publishTime", new Date());
        } else {
            condition.put("publishTime", DEFAULT_NULL_TIME);
        }
        int succ = this.wxNewsDao.update(condition, "status", "updateTime", "publishTime");
        if (succ != ids.size()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "参数错误");
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveContact(Long orgId, String richContent, Long coverId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        WxContactInfo contactInfo = this.wxContactDao.getByOrgId(orgId);
        if (contactInfo == null) {
            contactInfo = new WxContactInfo();
            contactInfo.setOrgId(orgId);
            contactInfo.setCreateTime(new Date());
        }
        if (coverId == null && richContent == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR);
        } else if (coverId == null) {
            // 更改文本内容
            if (richContent.length() > CONTACT_MAX_LENGTH) {
                throw new BussinessException(CommonErrorCode.PARAM_ERROR, "内容输入过长");
            }
            contactInfo.setRichContent(richContent);
        } else if (richContent == null) {
            // 更改banner图
            contactInfo.setCoverId(coverId);
        }
        contactInfo.setUpdateTime(new Date());
        this.wxContactDao.saveOrUpdateWithDefaultVal(contactInfo);
        return contactInfo.getId();
    }

    @Override
    public Map<String, Object> getContact(Long orgId) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "机构id错误");
        WxContactInfo contactInfo = this.wxContactDao.getByOrgId(orgId);
        Map<String, Object> data = new HashMap<>();
        Long coverId = 0l;
        String coverUrl = TianXiaoConstant.DEFAULT_CONTACT_BANNER;
        String info = "";
        if (contactInfo != null) {
            if (contactInfo.getCoverId() != null && contactInfo.getCoverId() > 0) {
                coverId = contactInfo.getCoverId();
                Map<Long, String> map = this.cacheCoverUrl(Sets.newHashSet(contactInfo.getCoverId()));
                if (map != null && map.containsKey(contactInfo.getCoverId())) {
                    coverUrl = map.get(contactInfo.getCoverId());
                }
            }
            if (StringUtils.isNotBlank(contactInfo.getRichContent())) {
                info = contactInfo.getRichContent();
            }
        }
        data.put("coverId", coverId);
        data.put("coverUrl", coverUrl);
        data.put("info", info);
        return data;
    }

    /**
     * 缓存新闻分类
     * @param orgId
     * @return
     */
    private Map<Long, WxCategory> cacheNewsCategory(Long orgId) {
        Map<Long, WxCategory> cache = new HashMap<>();
        List<WxCategory> categoryList = this.wxCategoryDao.getListByParam(orgId, CategoryType.NEWS.getCode(), null);
        if (CollectionUtils.isNotEmpty(categoryList)) {
            cache = BaseUtils.listToMap(categoryList, "id");
        }
        return cache;
    }

    /**
     * 缓存封面图片 OrgStorage
     * @param ids
     * @return
     */
    private Map<Long, String> cacheCoverUrl(Collection<Long> ids) {
        Map<Long, String> cache = new HashMap<>();
        if (CollectionUtils.isNotEmpty(ids)) {
            Map<Long, OrgStorage> storageMap = this.orgStorageDao.getOrgStorageMapByIds(ids);
            if (storageMap != null && storageMap.size() > 0) {
                for (Long id : ids) {
                    if (storageMap.containsKey(id)) {
                        OrgStorage storage = storageMap.get(id);
                        cache.put(id, StorageUtil.constructUrl(storage.getFid(), storage.getSn(), storage.getMimeType()));
                    } else {
                        cache.put(id, TianXiaoConstant.DEFAULT_NEWS_COVER);
                    }
                }
            }
        }
        return cache;
    }

    /**
     * 缓存操作人
     * @param ids
     * @return
     */
    private Map<Integer, String> cacheOperator(Long orgId, Collection<Integer> ids) {
        Map<Integer, String> cache = new HashMap<>();
        OrgInfo orgInfo = this.orgInfoDao.getOrgInfo(orgId.intValue(), "id", "contacts");
        if (orgInfo == null) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "机构信息不存在");
        }
        cache.put(0, orgInfo.getContacts());
        if (CollectionUtils.isNotEmpty(ids)) {
            Map<Long, String> map = this.txCascadeCredentialDao.getTxCascadCredentialListByCascdeIds(ids);
            if (map != null && map.size() > 0) {
                for (Long id : map.keySet()) {
                    cache.put(id.intValue(), map.get(id));
                }
            }
        }
        return cache;
    }

    @Override
    public List<WxNewsBaseDto> getBaseListByIds(Long orgId, Collection<Long> newsIds) {
        
        Map<String,Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("id", newsIds);
        condition.put("status", 1);
        condition.put("isDel", 0);
        
        List<WxNews> list = wxNewsDao.queryByCondition(condition, null);
        List<WxNewsBaseDto> data = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(list)) {
            Set<Long> coverIds = new HashSet<>();
            Set<Integer> operatorIds = new HashSet<>();
            for (WxNews po : list) {
                if (po.getCoverId() != null && po.getCoverId() > 0) {
                    coverIds.add(po.getCoverId());
                }
                if (po.getOperatorId() != null && po.getOperatorId() >= 0) {
                    operatorIds.add(po.getOperatorId());
                }
            }
            Map<Long, WxCategory> categoryMap = this.cacheNewsCategory(orgId);
            Map<Long, String> coverMap = null;
            if (CollectionUtils.isNotEmpty(coverIds)) {
                coverMap = this.cacheCoverUrl(coverIds);
            }
            Map<Integer, String> operatorMap = null;
            if (CollectionUtils.isNotEmpty(operatorIds)) {
                operatorMap = this.cacheOperator(orgId, operatorIds);
            }
            for (WxNews po : list) {
                WxNewsBaseDto dto = WxNewsDto.getFromPo(po, categoryMap, coverMap, operatorMap);
                data.add(dto);
            }
        }
        return data;
    }
    
    @Override
    public Map<Long,WxNewsBaseDto> getBaseMapByIds(Long orgId, Collection<Long> newsIds) {
        
        List<WxNewsBaseDto> list = this.getBaseListByIds(orgId, newsIds);
        
        Map<Long,WxNewsBaseDto> map = Maps.newHashMap();
        for(WxNewsBaseDto dto:list){
            map.put(dto.getId(), dto);
        }
        return map;
    }
    
    @Override
    public List<WxNewsBaseDto> getBaseListOrderByIds(Long orgId, Collection<Long> newsIds) {
        
        Map<Long,WxNewsBaseDto> maps = getBaseMapByIds(orgId,newsIds);
        
        List<WxNewsBaseDto> list = Lists.newArrayList();
        for(Long newsId:newsIds){
            WxNewsBaseDto dto = maps.get(newsId);
            if(dto!=null){
                list.add(dto);
            }
        }
        return list;
    }
}
