
/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2017 All Rights Reserved.
 */

package com.baijia.tianxiao.sal.wx.impl;

import com.baijia.tianxiao.consants.DataStatus;
import com.baijia.tianxiao.dal.enums.CourseTypeEnum;
import com.baijia.tianxiao.dal.org.dao.OrgClassLessonDao;
import com.baijia.tianxiao.dal.org.dao.OrgCourseDao;
import com.baijia.tianxiao.dal.comment.dao.OrgLessonCommentDao;
import com.baijia.tianxiao.dal.org.dao.OrgLessonSignDao;
import com.baijia.tianxiao.dal.org.dto.DeleteStatus;
import com.baijia.tianxiao.dal.org.po.*;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.comment.constant.CommentErrorCode;
import com.baijia.tianxiao.sal.course.constant.ArrangeStatus;
import com.baijia.tianxiao.sal.course.service.CourseStudentService;
import com.baijia.tianxiao.sal.course.service.CourseTeacherService;
import com.baijia.tianxiao.sal.course.service.OrgLessonConflictService;
import com.baijia.tianxiao.sal.teacher.api.TeacherService;
import com.baijia.tianxiao.sal.teacher.dto.OrgTeacherResponseDto;
import com.baijia.tianxiao.sal.wx.api.*;
import com.baijia.tianxiao.sal.wx.enums.RoomStatus;
import com.baijia.tianxiao.sal.wx.model.WxCourseLessonDto;
import com.baijia.tianxiao.sal.wx.model.WxLiveLessonView;
import com.baijia.tianxiao.sal.wx.model.WxVideoLessonView;
import com.baijia.tianxiao.sqlbuilder.bean.Order;
import com.baijia.tianxiao.sqlbuilder.dto.PageDto;
import com.baijia.tianxiao.util.BaseUtils;
import com.baijia.tianxiao.util.date.DateUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * @title WxLessonService
 * @desc TODO
 * @author leiruiqi
 * @date 2017年1月6日
 * @version 1.0
 */

@Slf4j
@Service("wxLessonService")
public class WxLessonServiceImpl implements WxLessonService {

    @Resource
    private OrgCourseDao orgCourseDao;

    @Resource
    private OrgClassLessonDao orgClassLessonDao;

    @Resource
    private WxLiveRoomLessonService wxLiveRoomLessonService;

    @Resource
    private WxVideoLessonService wxVideoLessonService;

    @Resource
    private CourseTeacherService courseTeacherService;

    @Resource
    private OrgLessonConflictService orgLessonConflictService;

    @Resource
    private OrgLessonCommentDao orgLessonCommentDao;

    @Resource
    private OrgLessonSignDao orgLessonSignDao;

    @Resource
    private CourseStudentService courseStudentService;

    @Resource
    private WxLiveRoomService wxLiveRoomService;

    @Resource
    private WxCloudSourceService wxCloudSourceService;

    @Autowired
    private TeacherService teacherService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addLiveClassLesson(Long orgId, WxCourseLessonDto dto) {

        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(dto != null, "class lessons param is illegal");
        Preconditions.checkArgument(dto.getCourseId() != null && dto.getCourseId() > 0, "courseId is illegal");
        Preconditions.checkArgument(dto.getTeacherId() != null && dto.getTeacherId() > 0, "teacherId is illegal");
        Preconditions.checkArgument(dto.getLessonName() != null && dto.getLessonName().length() < 21,
            "the lesson name is null or too long");
        Preconditions.checkArgument(dto.getStartTime() != null && dto.getEndTime() != null, "time is null");

        OrgCourse orgCourse = orgCourseDao.getByCourseId(dto.getCourseId());

        if (orgCourse == null || orgCourse.getCourseType() != CourseTypeEnum.COURSE_TYPE_CLASS_LIVE.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节类型参数错误");
        }
        if (orgCourse.getFreq() == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程未设课节数");
        }
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", orgCourse.getId());
        condition.put("delStatus", DeleteStatus.NORMAL.getValue());
        List<OrgClassLesson> allLessons = orgClassLessonDao.queryByCondition(condition, null);
        if (allLessons.size() >= orgCourse.getFreq()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "排课课节超过课程设置");
        }
        OrgClassLesson lesson = new OrgClassLesson();

        lesson.setStartTime(dto.getStartTime());
        lesson.setEndTime(dto.getEndTime());
        lesson.setCourseId(dto.getCourseId());
        lesson.setCreateTime(new Date());
        lesson.setDelStatus(DataStatus.NORMAL.getValue());
        lesson.setLayoutId(0l);
        lesson.setRoomId(0l);
        lesson.setUpdateTime(new Date());
        lesson.setNumber(0);
        lesson.setName(dto.getLessonName());
        lesson.setOrgId(orgId);

        orgClassLessonDao.save(lesson, "startTime", "endTime", "courseId", "createTime", "delStatus", "layoutId",
            "roomId", "updateTime", "orgId", "name");

        Set<Long> teacherIds = Sets.newHashSet();
        Set<Long> tempTeacherIds = Sets.newHashSet();
        if (dto.getTeacherId() != null && dto.getTeacherId() > 0) {
            teacherIds.add(dto.getTeacherId());
            tempTeacherIds.addAll(teacherIds);
            courseTeacherService.addTeacherToCourse(orgId, lesson.getCourseId(), teacherIds);// 先添加到课程
            courseTeacherService.addTeacherToLesson(orgId, lesson.getCourseId(), lesson.getId(), tempTeacherIds);
        }
        /**
         * 如果是新排课，设置cdb,org_course.arrange_lesson=1
         */
        if (orgCourse.getArrangeLesson() == ArrangeStatus.UNARRANGE.getCode()) {
            orgCourse.setArrangeLesson(ArrangeStatus.ARRANGEED.getCode());
            orgCourseDao.update(orgCourse, "arrangeLesson");
        }

        wxLiveRoomLessonService.create(lesson, orgCourse.getMaxStudent(), dto);

        orgClassLessonDao.update(lesson, "roomId");
        // 更新冲突检测表
        OrgLessonConflict conflict = new OrgLessonConflict();
        conflict.setOrgId(orgId);

        conflict.setTeacherId(dto.getTeacherId());
        //
        conflict.setRoomId(lesson.getRoomId());
        conflict.setLessonId(lesson.getId());
        conflict.setStartTime(lesson.getStartTime());
        conflict.setEndTime(lesson.getEndTime());
        conflict.setDelStatus(DataStatus.NORMAL.getValue());
        orgLessonConflictService.updateOrgLessonConflictsForce(conflict);

        resetLessonNumber(orgId, lesson.getCourseId());

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editLiveClassLesson(Long orgId, WxCourseLessonDto dto) {

        Preconditions.checkArgument(dto.getLessonId() != null && dto.getLessonId() > 0, "lessonId is illegal");
        Preconditions.checkArgument(dto.getTeacherId() != null && dto.getTeacherId() > 0, "teacherId is illegal");
        Preconditions.checkArgument(dto.getLessonName() != null && dto.getLessonName().length() < 21,
            "the lesson name is null or too long");
        OrgClassLesson lesson = orgClassLessonDao.getById(dto.getLessonId());

        OrgCourse orgCourse = orgCourseDao.getByCourseId(lesson.getCourseId());
        if (orgCourse == null || orgCourse.getCourseType() != CourseTypeEnum.COURSE_TYPE_CLASS_LIVE.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节类型参数错误");
        }

        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", dto.getLessonId(), lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }

        lesson.setDelStatus(DataStatus.NORMAL.getValue());
        boolean needRearrange = false;
        if (!dto.getEndTime().equals(lesson.getEndTime()) || !dto.getStartTime().equals(lesson.getStartTime())) {
            lesson.setEndTime(dto.getEndTime());
            lesson.setStartTime(dto.getStartTime());
            // 需要重新设置number
            needRearrange = true;
        }
        List<Long> teacherIds = Lists.newArrayList();

        if (dto.getTeacherId() == null) {
            dto.setTeacherId(0l);
        } else {
            teacherIds.add(dto.getTeacherId());
        }

        lesson.setUpdateTime(new Date());
        lesson.setName(dto.getLessonName());

        orgClassLessonDao.update(lesson);
        if (needRearrange) {
            resetLessonNumber(orgId, lesson.getCourseId());
        }

        courseTeacherService.resetLessonTeacher(orgId, lesson.getCourseId(), lesson.getId(), teacherIds);

        // 更新冲突检测表
        OrgLessonConflict conflict = new OrgLessonConflict();
        conflict.setOrgId(orgId);
        conflict.setTeacherId(dto.getTeacherId());
        conflict.setRoomId(lesson.getRoomId());
        conflict.setLessonId(dto.getLessonId());
        conflict.setStartTime(lesson.getStartTime());
        conflict.setEndTime(lesson.getEndTime());
        conflict.setDelStatus(DataStatus.NORMAL.getValue());
        orgLessonConflictService.updateOrgLessonConflictsForce(conflict);

        WxLiveRoom room = wxLiveRoomService.getLessonRoom(lesson.getOrgId(), lesson.getId());
        if (room == null) {
            wxLiveRoomLessonService.create(lesson, orgCourse.getMaxStudent(), dto);
        } else {
            lesson.setRoomId(room.getRoomId());
            wxLiveRoomLessonService.update(lesson, orgCourse.getMaxStudent(), dto);
        }

        if (needRearrange) {
            resetLessonNumber(orgId, lesson.getCourseId());
        }

    }

    private void resetLessonNumber(Long orgId, Long courseId) {
        List<OrgClassLesson> lessons = orgClassLessonDao.queryLessons(orgId, Lists.newArrayList(courseId), null, null,
            null, null, null, null, "id");
        if (CollectionUtils.isNotEmpty(lessons)) {
            int index = 1;
            @SuppressWarnings("unchecked")
            Map<String, Number>[] params = new Map[lessons.size()];
            for (OrgClassLesson lesson : lessons) {
                Map<String, Number> param = new HashMap<>();
                param.put("id", lesson.getId());
                param.put("number", index);
                params[index - 1] = param;
                index++;
            }
            orgClassLessonDao.batchUpdateLessonNumber(params);
        }
    }

    @Transactional
    @Override
    public void deleteLiveClassLesson(Long orgId, Long lessonId) {

        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", lessonId, lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节不存在或已被删除");
        }

        if (CollectionUtils.isNotEmpty(orgLessonCommentDao.getLessonComments(lessonId, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_COMMENT);
        }
        if (CollectionUtils.isNotEmpty(
            orgLessonSignDao.getLessonSignList(orgId, null, lessonId, null, null, null, true, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_SIGN);
        }
        courseStudentService.deleteStudentFromLesson(orgId, lessonId, null);
        courseTeacherService.delTeacherFromLesson(orgId, lessonId, null);

        List<Long> lessonIds = Lists.newArrayList();
        lessonIds.add(lessonId);

        orgClassLessonDao.updateDelByLessonIds(orgId, lessonIds);
        resetLessonNumber(orgId, lesson.getCourseId());

        // 冲突检测表逻辑删除
        orgLessonConflictService.delByLessonIds(orgId, lessonIds);

        wxLiveRoomLessonService.delete(lesson);
    }

    @Transactional
    @Override
    public void addVideoClassLesson(Long orgId, WxCourseLessonDto dto) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(dto != null, "class lessons param is illegal");
        Preconditions.checkArgument(dto.getCourseId() != null && dto.getCourseId() > 0, "courseId is illegal");
        Preconditions.checkArgument(dto.getLessonName() != null && dto.getLessonName().length() < 21,
            "the lesson name is null or too long");
        Preconditions.checkArgument(dto.getSourceId() != null, "sourcdId not exit");
        OrgCourse orgCourse = orgCourseDao.getByCourseId(dto.getCourseId());
        Preconditions.checkArgument(orgCourse != null, "course not exit");

        if (orgCourse.getCourseType() != CourseTypeEnum.COURSE_TYPE_CLASS_VIDEO.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节类型参数错误");
        }

        if (orgCourse.getFreq() == null) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课程未设课节数");
        }
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", orgCourse.getId());
        condition.put("delStatus", DeleteStatus.NORMAL.getValue());
        List<OrgClassLesson> allLessons = orgClassLessonDao.queryByCondition(condition, null);
        if (allLessons.size() >= orgCourse.getFreq()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "排课课节超过课程设置");
        }

        OrgClassLesson lesson = new OrgClassLesson();
        Date now = new Date();
        lesson.setStartTime(now);
        lesson.setEndTime(now);
        lesson.setCourseId(dto.getCourseId());
        lesson.setCreateTime(now);
        lesson.setDelStatus(DataStatus.NORMAL.getValue());
        lesson.setLayoutId(0l);
        lesson.setRoomId(0l);
        lesson.setUpdateTime(now);
        lesson.setNumber(0);
        lesson.setName(dto.getLessonName());
        lesson.setOrgId(orgId);

        orgClassLessonDao.save(lesson, "startTime", "endTime", "courseId", "createTime", "delStatus", "layoutId",
            "roomId", "updateTime", "orgId", "name");

        dto.setLessonId(lesson.getId());
        dto.setOrgId(orgId);
        this.wxVideoLessonService.saveByWxCourseLessonDto(dto);

    }

    @Transactional
    @Override
    public void editVideoClassLesson(Long orgId, WxCourseLessonDto dto) {
        dto.setOrgId(orgId);
        OrgClassLesson lesson = orgClassLessonDao.getById(dto.getLessonId());

        OrgCourse orgCourse = orgCourseDao.getByCourseId(dto.getCourseId());
        Preconditions.checkArgument(orgCourse != null, "course not exit");

        if (orgCourse.getCourseType() != CourseTypeEnum.COURSE_TYPE_CLASS_VIDEO.getCode()) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节类型参数错误");
        }

        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", dto.getLessonId(), lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节ID不正确");
        }

        lesson.setDelStatus(DataStatus.NORMAL.getValue());

        List<Long> teacherIds = Lists.newArrayList();

        if (dto.getTeacherId() == null) {
            dto.setTeacherId(0l);
        } else {
            teacherIds.add(dto.getTeacherId());
        }

        lesson.setUpdateTime(new Date());
        lesson.setName(dto.getLessonName());

        orgClassLessonDao.update(lesson);

        dto.setLessonId(lesson.getId());

        this.wxVideoLessonService.updateByWxCourseLessonDto(dto);
    }

    @Transactional
    @Override
    public void deleteVideoClassLesson(Long orgId, Long lessonId) {

        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        Preconditions.checkArgument(lessonId != null && lessonId > 0, "lessonId is illegal");
        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        if (lesson == null || !orgId.equals(lesson.getOrgId())) {
            log.warn("lesson id:{} is illegal,query lesson:{}", lessonId, lesson);
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "课节不存在或已被删除");
        }

        if (CollectionUtils.isNotEmpty(orgLessonCommentDao.getLessonComments(lessonId, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_COMMENT);
        }
        if (CollectionUtils.isNotEmpty(
            orgLessonSignDao.getLessonSignList(orgId, null, lessonId, null, null, null, true, null, null, "id"))) {
            throw new BussinessException(CommentErrorCode.LESSON_WITH_SIGN);
        }
        courseStudentService.deleteStudentFromLesson(orgId, lessonId, null);
        courseTeacherService.delTeacherFromLesson(orgId, lessonId, null);

        List<Long> lessonIds = Lists.newArrayList();
        lessonIds.add(lessonId);

        orgClassLessonDao.updateDelByLessonIds(orgId, lessonIds);

        this.wxVideoLessonService.delByLessonId(orgId, lessonId);

    }

    @Override
    public List<WxLiveLessonView> listLiveClassLesson(Long orgId, Long courseId, PageDto pageDto) {

        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        if (courseId != null) {
            condition.put("courseId", courseId);
        }

        condition.put("delStatus", 0);
        List<OrgClassLesson> lessonList =
            this.orgClassLessonDao.queryByCondition(condition, Order.asc("startTime"), pageDto);
        List<Long> lessonIds = Lists.newArrayList();
        for (OrgClassLesson lesosn : lessonList) {
            lessonIds.add(lesosn.getId());
        }
        List<WxLiveLessonView> wxLiveList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(lessonIds)) {
            return wxLiveList;
        }
        Map<Long, WxLiveRoom> liveRoomMap = this.wxLiveRoomService.queryMapByOrgLessonIds(orgId, lessonIds);

        Set<Long> teacherIdSet = Sets.newHashSet();
        for(WxLiveRoom room : liveRoomMap.values()){
            teacherIdSet.add(room.getAdminId());
            teacherIdSet.add(room.getTeacherId());
        }
        Map<Long, OrgTeacherResponseDto> teacherMap = teacherService.getOrgTeacherMap(orgId, teacherIdSet);
        
        Date now = new Date();
        for (OrgClassLesson lesson : lessonList) {
            WxLiveRoom liveLesson = liveRoomMap.get(lesson.getId());
            WxLiveLessonView view = buildWxLiveLessonView(lesson, liveLesson, now, teacherMap);
            wxLiveList.add(view);
        }
        return wxLiveList;

    }

    @Override
    public List<WxVideoLessonView> listVideoClassLesson(Long orgId, Long courseId) {

        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseId);
        condition.put("delStatus", 0);
        List<OrgClassLesson> lessonList = this.orgClassLessonDao.queryByCondition(condition, Order.asc("id"), null);
        List<Long> lessonIds = Lists.newArrayList();
        for (OrgClassLesson lesosn : lessonList) {
            lessonIds.add(lesosn.getId());
        }
        Map<Long, WxVideoLesson> wxVideoMap = wxVideoLessonService.queryMapByOrgLessonId(orgId, lessonIds);
        List<Long> sourceIds = Lists.newArrayList();
        for (WxVideoLesson videoLesson : wxVideoMap.values()) {
            sourceIds.add(videoLesson.getSourceId());
        }

        Map<Long, WxCloudSource> cloudSourceMap = wxCloudSourceService.getMapByIds(orgId, sourceIds);
        List<WxVideoLessonView> wxVideoList = Lists.newArrayList();
        for (OrgClassLesson lesson : lessonList) {
            WxVideoLesson videoLesson = wxVideoMap.get(lesson.getId());
            WxCloudSource cloudSource = null;
            if (videoLesson != null) {
                cloudSource = cloudSourceMap.get(videoLesson.getSourceId());
            }
            WxVideoLessonView view = buildWxVideoLessonView(lesson, videoLesson, cloudSource);
            wxVideoList.add(view);
        }
        return wxVideoList;

    }

    private WxVideoLessonView buildWxVideoLessonView(OrgClassLesson lesson, WxVideoLesson video,
        WxCloudSource cloudSource) {
        WxVideoLessonView view = new WxVideoLessonView();

        view.setCourseId(lesson.getCourseId());
        view.setLessonId(lesson.getId());
        view.setName(lesson.getName());
        view.setNumber(lesson.getNumber());
        view.setOrgId(lesson.getOrgId());
        view.setCreateTime(lesson.getCreateTime());
        view.setUpdateTime(lesson.getUpdateTime());

        if (video != null) {
            view.setIsFree(video.getIsFree());
            view.setSourceId(video.getSourceId());

        }
        if (cloudSource != null) {
            view.setLength(cloudSource.getLength());
            view.setSourceName(cloudSource.getName());
            if (view.getLength() != null) {
                view.setLengthStr(DateUtil.secToTime(view.getLength().intValue()));
            }
        }

        return view;
    }

    private WxLiveLessonView buildWxLiveLessonView(OrgClassLesson lesson, WxLiveRoom liveRoom, Date now,
        Map<Long, OrgTeacherResponseDto> teacherMap) {
        WxLiveLessonView view = new WxLiveLessonView();
        view.setCourseId(lesson.getCourseId());
        view.setLessonId(lesson.getId());
        view.setName(lesson.getName());
        view.setNumber(lesson.getNumber());
        view.setOrgId(lesson.getOrgId());
        view.setCreateTime(lesson.getCreateTime());
        view.setUpdateTime(lesson.getUpdateTime());

        if (liveRoom != null) {
            view.setRoomId(liveRoom.getRoomId());
            view.setStartTime(liveRoom.getStartTime());
            view.setEndTime(liveRoom.getEndTime());
            view.setAdminId(liveRoom.getAdminId());
            view.setTeacherId(liveRoom.getTeacherId());
            view.setAdmin(teacherMap.get(liveRoom.getAdminId()));
            view.setAdminName(view.getAdmin().getTeacherName());
            view.setTeacher(teacherMap.get(liveRoom.getTeacherId()));
            view.setTeacherName(view.getTeacher().getTeacherName());

            boolean result = view.getStartTime().before(now);
            if (!result) {
                view.setRoomStatus(RoomStatus.NOT_BEGIN.getCode());
            } else {
                if (view.getEndTime().before(now)) {
                    view.setRoomStatus(RoomStatus.END.getCode());
                } else {
                    view.setRoomStatus(RoomStatus.ON_GOING.getCode());
                }
            }

            view.setPlayBack(liveRoom.getPlayBack());

        }

        return view;
    }

    @Override
    public WxLiveLessonView getLiveClassLesson(Long orgId, Long lessonId) {

        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        if (lesson == null || lesson.getOrgId().longValue() != orgId) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节id错误");
        }

        WxLiveRoom liveLesson = wxLiveRoomService.getLessonRoom(orgId, lessonId);
        Map<Long, OrgTeacherResponseDto> teacherMap = teacherService.getOrgTeacherMap(orgId,
            Lists.newArrayList(liveLesson.getAdminId(), liveLesson.getTeacherId()));

        WxLiveLessonView view = buildWxLiveLessonView(lesson, liveLesson, new Date(), teacherMap);
        return view;

    }

    @Override
    public WxVideoLessonView getVideoClassLesson(Long orgId, Long lessonId) {

        OrgClassLesson lesson = orgClassLessonDao.getById(lessonId);
        if (lesson == null || lesson.getOrgId().longValue() != orgId) {
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "课节id错误");
        }

        WxVideoLesson videoLesson = wxVideoLessonService.getByLessonId(orgId, lessonId);
        WxCloudSource cloudSource = null;
        if (videoLesson != null) {
            cloudSource = wxCloudSourceService.getById(orgId, videoLesson.getSourceId());
        }

        WxVideoLessonView view = buildWxVideoLessonView(lesson, videoLesson, cloudSource);
        return view;

    }

    @Override
    public List<WxVideoLessonView> getVideoClassList(Long orgId, Collection<Long> courseIds) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(courseIds)) {
            return Collections.emptyList();
        }
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("orgId", orgId);
        condition.put("courseId", courseIds);
        condition.put("delStatus", 0);
        List<OrgClassLesson> lessonList =
            this.orgClassLessonDao.queryByCondition(condition, Order.desc("number"), null);
        List<Long> lessonIds = Lists.newArrayList();
        for (OrgClassLesson lesosn : lessonList) {
            lessonIds.add(lesosn.getId());
        }

        Map<Long, WxVideoLesson> wxVideoMap = wxVideoLessonService.queryMapByOrgLessonId(orgId, lessonIds);
        List<Long> sourceIds = Lists.newArrayList();
        for (WxVideoLesson videoLesson : wxVideoMap.values()) {
            sourceIds.add(videoLesson.getSourceId());
        }

        Map<Long, WxCloudSource> cloudSourceMap = wxCloudSourceService.getMapByIds(orgId, sourceIds);
        List<WxVideoLessonView> wxVideoList = Lists.newArrayList();
        for (OrgClassLesson lesson : lessonList) {
            WxVideoLesson videoLesson = wxVideoMap.get(lesson.getId());
            WxCloudSource cloudSource = null;
            if (videoLesson != null) {
                cloudSource = cloudSourceMap.get(videoLesson.getSourceId());
            }
            WxVideoLessonView view = buildWxVideoLessonView(lesson, videoLesson, cloudSource);
            wxVideoList.add(view);
        }
        return wxVideoList;
    }

    @Override
    public List<WxVideoLessonView> getVideoLessonViewList(Long orgId, List<OrgClassLesson> lessons) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId is illegal");
        if (CollectionUtils.isEmpty(lessons)) {
            return Collections.emptyList();
        }
        Set<Long> lessonIds = BaseUtils.getPropertiesList(lessons, "id");

        Map<Long, WxVideoLesson> wxVideoMap = wxVideoLessonService.queryMapByOrgLessonId(orgId, lessonIds);
        List<Long> sourceIds = Lists.newArrayList();
        for (WxVideoLesson videoLesson : wxVideoMap.values()) {
            sourceIds.add(videoLesson.getSourceId());
        }

        Map<Long, WxCloudSource> cloudSourceMap = wxCloudSourceService.getMapByIds(orgId, sourceIds);
        List<WxVideoLessonView> wxVideoList = Lists.newArrayList();
        for (OrgClassLesson lesson : lessons) {
            WxVideoLesson videoLesson = wxVideoMap.get(lesson.getId());
            WxCloudSource cloudSource = null;
            if (videoLesson != null) {
                cloudSource = cloudSourceMap.get(videoLesson.getSourceId());
            }
            WxVideoLessonView view = buildWxVideoLessonView(lesson, videoLesson, cloudSource);
            wxVideoList.add(view);
        }
        return wxVideoList;
    }

    /*
     * @Override public List<WxLiveLessonView> queryLiveClassLesson(WxCourseLessonDto dto, PageDto pageDto) {
     * 
     * Map<String,Object> condition = Maps.newHashMap(); if(dto.getOrgId()!=null) condition.put("orgId", orgId); }
     * 
     * if(courseId!=null){ condition.put("courseId",courseId); }
     * 
     * condition.put("delStatus",0); List<OrgClassLesson> lessonList =
     * this.orgClassLessonDao.queryByCondition(condition, Order.desc("startTime"), pageDto); List<Long> lessonIds =
     * Lists.newArrayList(); for(OrgClassLesson lesosn:lessonList){ lessonIds.add(lesosn.getId()); }
     * Map<Long,WxLiveRoom> liveRoomMap = this.wxLiveRoomService.queryMapByOrgLessonIds(orgId, lessonIds);
     * 
     * List<WxLiveLessonView> wxLiveList = Lists.newArrayList(); for(OrgClassLesson lesson:lessonList){ WxLiveRoom
     * liveLesson = liveRoomMap.get(lesson.getId()); WxLiveLessonView view = buildWxLiveLessonView(lesson,liveLesson);
     * wxLiveList.add(view); } return wxLiveList;
     * 
     * }
     */

}
