package com.baijia.tianxiao.sal.wx.impl;

import com.baijia.tianxiao.constants.CategoryType;
import com.baijia.tianxiao.dal.org.constant.DeleteStatus;
import com.baijia.tianxiao.dal.org.dao.WxCategoryDao;
import com.baijia.tianxiao.dal.org.dao.WxNewsDao;
import com.baijia.tianxiao.dal.org.po.WxCategory;
import com.baijia.tianxiao.enums.CommonErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.sal.wx.api.WxCategoryService;
import com.baijia.tianxiao.sal.wx.enums.WxErrorCode;
import com.baijia.tianxiao.sal.wx.model.WxCategoryDto;
import com.baijia.tianxiao.util.BaseUtils;

import com.google.common.base.Preconditions;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Created by wangsixia on 2017/1/13.
 */
@Service
public class WxCategoryServiceImpl implements WxCategoryService {

    @Autowired
    private WxCategoryDao wxCategoryDao;
    @Autowired
    private WxNewsDao wxNewsDao;

    @Override
    @Transactional(rollbackFor = { Exception.class} )
    public Long add(Long orgId, WxCategoryDto param) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId 错误");
        Preconditions.checkArgument(param != null && StringUtils.isNotBlank(param.getName()), "分类名不能为空");
        Preconditions.checkArgument(CategoryType.getBycode(param.getType()) != null, "类型参数错误");

        int count = this.wxCategoryDao.checkExist(orgId, param.getType(), param.getName());
        if (count > 0) {
            throw new BussinessException(WxErrorCode.CATEGORY_NAME_EXIST);
        }

        List<WxCategory> list = this.wxCategoryDao.getListByParam(orgId, param.getType(), null);
        int size = (CollectionUtils.isNotEmpty(list) ? list.size() : 0);

        WxCategory po = new WxCategory();
        po.setName(param.getName());
        po.setDisplayOrder(size + 1);
        po.setOrgId(orgId);
        po.setType(param.getType());
        po.setIsDel(DeleteStatus.NORMAL.getValue());
        po.setCreateTime(new Date());
        po.setUpdateTime(new Date());
        this.wxCategoryDao.save(po);
        return po.getId();
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class} )
    public void delete(Long orgId, Long id) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId 错误");
        Preconditions.checkArgument(id != null && id > 0, "分类id错误");
        WxCategory po = this.wxCategoryDao.getById(id);
        if (po == null || po.getIsDel().intValue() == DeleteStatus.DELETED.getValue()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "分类不存在或已被删除");
        }
        if (po.getOrgId().longValue() != orgId.longValue()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "分类不属于该机构");
        }
        List<?> list = wxNewsDao.listByOrgIdCategoryId(orgId, id);
        if (CollectionUtils.isNotEmpty(list)){
            throw new BussinessException(CommonErrorCode.BUSINESS_ERROR, "该分类已被"+list.size()+"个新闻使用，无法删除");
        }
        po.setIsDel(DeleteStatus.DELETED.getValue());
        po.setUpdateTime(new Date());
        this.wxCategoryDao.update(po, "isDel", "updateTime");
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class} )
    public void mod(Long orgId, Long id, String name) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId 错误");
        Preconditions.checkArgument(StringUtils.isNotBlank(name) && id != null && id > 0, "分类参数错误");
        WxCategory po = this.wxCategoryDao.getById(id);
        if (po == null || po.getIsDel().intValue() == DeleteStatus.DELETED.getValue()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "分类不存在或已被删除");
        }
        if (po.getOrgId().longValue() != orgId.longValue()) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "分类不属于该机构");
        }
        int count = this.wxCategoryDao.checkExist(orgId, po.getType(), name);
        if (count > 0 && !po.getName().equals(name)) {
            throw new BussinessException(CommonErrorCode.PARAM_ERROR, "分类名已存在");
        }
        if (!po.getName().equals(name)) {
            po.setName(name);
            po.setUpdateTime(new Date());
            this.wxCategoryDao.update(po, "name", "updateTime");
        }
    }

    @Override
    public List<WxCategoryDto> getCategoryList(Long orgId, Integer type) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId 错误");
        Preconditions.checkArgument(CategoryType.getBycode(type) != null, "分类类型错误");
        List<WxCategoryDto> data = new ArrayList<>();
        List<WxCategory> list = this.wxCategoryDao.getListByParam(orgId, type, null);
        if (CollectionUtils.isNotEmpty(list)) {
            for (WxCategory po : list) {
                WxCategoryDto dto = new WxCategoryDto();
                dto.setId(po.getId());
                dto.setName(po.getName());
                dto.setType(type);
                data.add(dto);
            }
        }
        return data;
    }

    @Override
    @Transactional(rollbackFor = { Exception.class, BussinessException.class} )
    public void order(Long orgId, Integer type, List<Long> ids) {
        Preconditions.checkArgument(orgId != null && orgId > 0, "orgId 错误");
        Preconditions.checkArgument(CategoryType.getBycode(type) != null, "分类类型错误");
        if (CollectionUtils.isNotEmpty(ids)) {
            List<WxCategory> list = this.wxCategoryDao.getListByParam(orgId, type, ids);
            if (CollectionUtils.isNotEmpty(list)) {
                Map<Long, WxCategory> map = BaseUtils.listToMap(list, "id");
                int size = list.size();
                for (Long id : ids) {
                    WxCategory po = map.get(id);
                    if (po != null) {
                        po.setDisplayOrder(size--);
                        po.setUpdateTime(new Date());
                        this.wxCategoryDao.update(po, "displayOrder", "updateTime");
                    }
                }
            }
        }
    }
}
