/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.wechat.po.ComponentVerifyTicket;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationReceiveService;
import com.baijia.tianxiao.sal.wechat.api.BindingService;
import com.baijia.tianxiao.sal.wechat.api.ComponentVerifyTicketService;
import com.baijia.tianxiao.sal.wechat.api.UnifiedWechatAccountService;
import com.baijia.tianxiao.sal.wechat.dto.request.AuthorizationReceiveDto;
import com.baijia.tianxiao.sal.wechat.dto.request.AuthorizationReceiveRequest;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.common.WXBizMsgCrypt;
import com.baijia.tianxiao.sal.wechat.helper.common.XMLParse;

/**
 * @title AuthorizationReceiveServiceImpl
 * @desc 微信第三方平台 “授权事件接收URL” 服务类  (接收component_verify_ticket和解除授权事件)
 * @author zhenyujian
 * @date 2015年8月5日
 * @version 1.0
 */
@Slf4j
@Service
public class AuthorizationReceiveServiceImpl implements AuthorizationReceiveService {

    @Autowired
    private BindingService bindingService;
    @Autowired
    private ComponentVerifyTicketService componentVerifyTicketService;

    
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle(AuthorizationReceiveRequest request) {
    	try {
	        //消息解密
	        String appId = WechatProperties.getOpenAppId();
	        String token = WechatProperties.getOpenMsgToken();
	        String encodingAesKey = WechatProperties.getEncodingAesKey();
	        WXBizMsgCrypt wxBizMsgCrypt = new WXBizMsgCrypt(token, encodingAesKey, appId);
	        String[] encrytMsg = XMLParse.extractTicket(request.getPostData());

	        String decryptXmlMsg = wxBizMsgCrypt.decryptMsg(request.getMsgSignature(), request.getTimestamp(), request.getNonce(), encrytMsg);	        
	        AuthorizationReceiveDto dto = AuthorizationReceiveDto.buildByXml(decryptXmlMsg);
	        
	        switch(dto.getInfoType()) {
	        	case "component_verify_ticket" :
	        		handleVerifyTicket(dto);
	        		break;
	        		
	        	case "unauthorized" :
	        		handleUnauthorization(dto);
		            return;
	        }
	        
    	} catch (Exception e) {
    	    log.error("AuthorizationReceiveRequest ={}",e);
			e.printStackTrace();
		}
    }

    //接收微信服务器定时发送的 component_verify_ticket
    private void handleVerifyTicket(AuthorizationReceiveDto dto) {
        log.info("wechat - AuthorizationReceiveServiceImpl - handleVerifyTicket - data:{}", dto);
        ComponentVerifyTicket newTicket = new ComponentVerifyTicket();
        newTicket.setAppId(dto.getAppId());
        newTicket.setInfoType(dto.getInfoType());
        newTicket.setComponentVerifyTicket(dto.getComponentVerifyTicket());
        newTicket.setCreateTime(dto.getCreateTime());
        componentVerifyTicketService.saveOrUpdate(newTicket);
    }

    //接收用户取消授权事件
    private void handleUnauthorization(AuthorizationReceiveDto dto) {
        log.info("wechat - AuthorizationReceiveServiceImpl - handleUnauthorization - data:{}", dto);
        bindingService.removeBindingInfo(dto.getAuthorizerAppid());
    }
    
}
