/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.helper;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

import com.baijia.commons.lang.utils.http.HttpClientUtils;
import com.baijia.tianxiao.sal.wechat.constant.MediaType;
import com.baijia.tianxiao.sal.wechat.constant.WechatErrorCode;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.WebServiceException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.enums.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.WechatApiResponse;

/**
 * @Title : WechatRemoteCallUtils
 * @Description :
 * @Author : zhenyujian
 * @Date : 2015年12月4日 上午11:00:28
 */
@Slf4j
public class WechatRemoteCallHelper {

    /**
     * @Description : GET请求 url
     * @Author : zhenyujian
     * @Date : 2015年12月12日 下午2:55:18
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse get(String url, boolean logResponse) throws WechatException, WebServiceException {
        try {
            String response = HttpClientUtils.doGet(url, null);
            if (logResponse) {
                log.info("WechatRemoteCallHelper - url:{}, response:{}", url, response);
            } else {
                log.info("WechatRemoteCallHelper - url:{}", url);
            }

            return convertJson(response);
        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - get - url:{}", url);
            log.warn("WechatRemoteCallHelper - get - Exception - ", e);
            throw new WebServiceException(e);
        }
    }

    public static WechatApiResponse get(String url) {
        return get(url, true);
    }

    /**
     * @Description : GET请求 url params
     * @Author : zhenyujian
     * @Date : 2015年12月12日 下午2:55:36
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse get(String url, Map<String, Object> params, boolean logResponse)
        throws WechatException, WebServiceException {
        try {
            if (params != null && !params.isEmpty()) {
                StringBuilder builder = new StringBuilder(url);
                if (!url.contains("?")) {
                    builder.append("?");
                }
                for (String key : params.keySet()) {
                    builder.append("&").append(key).append("=").append(params.get(key));
                }
                url = builder.toString();
            }

            String response = HttpClientUtils.doGet(url, null);
            if (logResponse) {
                log.info("WechatRemoteCallHelper - url:{}, response:{}", url, response);
            } else {
                log.info("WechatRemoteCallHelper - url:{}", url);
            }

            return convertJson(response);
        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - get - url:{}", url);
            log.warn("WechatRemoteCallHelper - get - exception - ", e);
            throw new WebServiceException(e);
        }
    }

    public static WechatApiResponse get(String url, Map<String, Object> params) {
        return get(url, params, true);
    }

    /**
     * @Description : POST请求 提交json
     * @Author : zhenyujian
     * @Date : 2015年12月28日 上午10:42:24
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse postJson(String url, Map<String, Object> params, boolean logResponse)
        throws WechatException, WebServiceException {
        try {
            String response = doPostJSON(url, JSONObject.fromObject(params).toString());
            log.info("WechatRemoteCallHelper - url:{}, params:{}, response:{}", url, params, response);
            if (logResponse) {
                log.info("WechatRemoteCallHelper - url:{}, params:{}, response:{}", url, params, response);
            } else {
                log.info("WechatRemoteCallHelper - url:{}, params:{}", url);
            }

            return convertJson(response);
        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - postJson - url:{}, params", url, params);
            log.warn("WechatRemoteCallHelper - postJson - Exception - ", e);
            throw new WebServiceException(e);
        }
    }

    public static WechatApiResponse postJson(String url, Map<String, Object> params) {
        return postJson(url, params, true);
    }

    /**
     * @Description : POST请求 提交json
     * @Author : zhenyujian
     * @Date : 2015年12月28日 上午10:47:31
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse postJsonObj(String url, JSONObject json, boolean logResponse)
        throws WechatException, WebServiceException {
        try {
            String response = doPostJSON(url, json.toString());
            if (logResponse) {
                log.info("WechatRemoteCallHelper - url:{}, params:{}, response:{}", url, json, response);
            } else {
                log.info("WechatRemoteCallHelper - url:{}, params:{}", url);
            }

            return convertJson(response);
        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - postJson - url:{}, params", url, json);
            log.warn("WechatRemoteCallHelper - postJson - Exception - ", e);
            throw new WebServiceException(e);
        }
    }

    public static WechatApiResponse postJsonObj(String url, JSONObject json) {
        return postJsonObj(url, json, true);
    }

    public static WechatApiResponse postJsonStr(String url, String jsonStr, boolean logResponse) {
        String response = doPostJSON(url, jsonStr);
        if (logResponse) {
            log.info("WechatRemoteCallHelper - url:{}, params:{}, response:{}", url, jsonStr, response);
        } else {
            log.info("WechatRemoteCallHelper - url:{}, params:{}", url);
        }
        return convertJson(response);
    }

    /**
     * @Description : POST 上传文件
     * @Author : zhenyujian
     * @Date : 2015年12月12日 下午2:55:51
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse postFile(String url, Map<String, String> params, File file)
        throws WechatException, WebServiceException {
        try {
            if (file == null) {
                throw new RuntimeException("postFile - file is null");
            }
            if (params == null) {
                params = new HashMap<String, String>();
            }
            Map<String, File> files = new HashMap<String, File>();
            files.put(file.getName(), file);

            String response = com.baijia.tianxiao.util.httpclient.HttpClientUtils.doPostFiles(url, params, files);
            log.info("request - url:{}, params:{}, fileName:{}, response:{}", url, params, file.getName(), response);

            return convertJson(response);

        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - postFile - url:{}, fileName:{}", url, file.getName());
            log.warn("WechatRemoteCallHelper - postFile - Exception - ", e);
            throw new WebServiceException(e);
        }
    }

    /**
     * @Description : POST 上传微信永久素材
     * @Author : zhenyujian
     * @Date : 2016年4月12日 下午5:10:19
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse postWechatPermanentMedia(String urlStr, Map<String, String> params, File file,
        MediaType mediaType) {
        try {
            URL url = new URL(urlStr);
            String result = null;
            long filelength = file.length();
            String fileName = file.getName();
            // String suffix = fileName.substring(fileName.lastIndexOf("."),fileName.length());
            String type = mediaType.getStr();
            /**
             * 你们需要在这里根据文件后缀suffix将type的值设置成对应的mime类型的值
             */
            HttpURLConnection con = (HttpURLConnection) url.openConnection();
            con.setRequestMethod("POST"); // 以Post方式提交表单，默认get方式
            con.setDoInput(true);
            con.setDoOutput(true);
            con.setUseCaches(false); // post方式不能使用缓存
            // 设置请求头信息
            con.setRequestProperty("Connection", "Keep-Alive");
            con.setRequestProperty("Charset", "UTF-8");

            // 设置边界,这里的boundary是http协议里面的分割符，不懂的可惜百度(http 协议 boundary)，这里boundary 可以是任意的值(111,2222)都行
            String BOUNDARY = "----------" + System.currentTimeMillis();
            con.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);
            // 请求正文信息
            // 第一部分：

            StringBuilder sb = new StringBuilder();

            // 这块是post提交type的值也就是文件对应的mime类型值
            sb.append("--"); // 必须多两道线 这里说明下，这两个横杠是http协议要求的，用来分隔提交的参数用的，不懂的可以看看http 协议头
            sb.append(BOUNDARY);
            sb.append("\r\n");
            sb.append("Content-Disposition: form-data;name=\"type\" \r\n\r\n"); // 这里是参数名，参数名和值之间要用两次
            sb.append(type + "\r\n"); // 参数的值

            // 这块是上传video是必须的参数，你们可以在这里根据文件类型做if/else 判断
            sb.append("--"); // 必须多两道线
            sb.append(BOUNDARY);
            sb.append("\r\n");
            sb.append("Content-Disposition: form-data;name=\"description\" \r\n\r\n");
            // sb.append(j.toString()+"\r\n");

            /**
             * 这里重点说明下，上面两个参数完全可以卸载url地址后面 就想我们平时url地址传参一样，
             * http://api.weixin.qq.com/cgi-bin/material/add_material?access_token=##ACCESS_TOKEN##&type=""&description=
             * {} 这样，如果写成这样，上面的 那两个参数的代码就不用写了，不过media参数能否这样提交我没有试，感兴趣的可以试试
             */

            sb.append("--"); // 必须多两道线
            sb.append(BOUNDARY);
            sb.append("\r\n");
            // 这里是media参数相关的信息，这里是否能分开下我没有试，感兴趣的可以试试
            sb.append("Content-Disposition: form-data;name=\"media\";filename=\"" + fileName + "\";filelength=\""
                + filelength + "\" \r\n");
            sb.append("Content-Type:application/octet-stream\r\n\r\n");
            // System.out.println(sb.toString());
            byte[] head = sb.toString().getBytes("utf-8");
            // 获得输出流
            OutputStream out = new DataOutputStream(con.getOutputStream());
            // 输出表头
            out.write(head);
            // 文件正文部分
            // 把文件已流文件的方式 推入到url中
            DataInputStream in = new DataInputStream(new FileInputStream(file));
            int bytes = 0;
            byte[] bufferOut = new byte[1024];
            while ((bytes = in.read(bufferOut)) != -1) {
                out.write(bufferOut, 0, bytes);
            }
            in.close();
            // 结尾部分，这里结尾表示整体的参数的结尾，结尾要用"--"作为结束，这些都是http协议的规定
            byte[] foot = ("\r\n--" + BOUNDARY + "--\r\n").getBytes("utf-8");// 定义最后数据分隔线
            out.write(foot);
            out.flush();
            out.close();
            StringBuffer buffer = new StringBuffer();
            BufferedReader reader = null;
            // 定义BufferedReader输入流来读取URL的响应
            reader = new BufferedReader(new InputStreamReader(con.getInputStream()));
            String line = null;
            while ((line = reader.readLine()) != null) {
                buffer.append(line);
            }
            if (result == null) {
                result = buffer.toString();
            }

            return convertJson(result);

        } catch (WechatException e) {
            throw e;
        } catch (Exception e) {
            log.warn("WechatRemoteCallHelper - postWechatPermanentMedia - url:{}, fileName:{}", urlStr, file.getName());
            log.warn("WechatRemoteCallHelper - postWechatPermanentMedia - Exception - ", e);
            throw new WebServiceException(e);
        }
    }

    /**
     * @Description : 解析 微信接口返回结果
     * @Author : zhenyujian
     * @Date : 2015年12月12日 下午2:59:05
     *
     * @Return : WechatApiResponse
     */
    public static WechatApiResponse convertJson(String jsonData) throws WechatException {
        int errorCode = 0;
        String errorMsg = "";

        if (StringUtils.isNotBlank(jsonData)) {
            JSONObject jsonObj = JSONObject.fromObject(jsonData);
            errorCode = jsonObj.optInt("errcode", 0);
            errorMsg = jsonObj.optString("errmsg", "");

            if (errorCode != 0) {

                // 返回码见: http://mp.weixin.qq.com/wiki/17/fa4e1434e57290788bde25603fa2fcbd.html
                // 针对需要特殊处理的errorCode进行转换 其他统一返回WECHAT_API_RETURN_ERROR
                // 此处判断不使用枚举类型 因为微信文档错误码不完整 此处使用实际调试时发现的需要处理的错误码
                if (errorCode == -1) {
                    // 微信服务器系统繁忙
                    throw new BussinessException(SalWechatErrorCode.WECHAT_SYSTEM_ERROR);
                } else if (errorCode == 40001 || errorCode == 40014 || errorCode == 42001) {
                    // accessToken无效
                    throw new WechatException(SalWechatErrorCode.WECHAT_ACCESS_TOKEN_INVALID);
                } else if (errorCode == 48001) {
                    // 公众号权限不足
                    throw new WechatException(SalWechatErrorCode.WECHAT_PERMISSION_DENIED);
                } else if (errorCode == 61007) {
                    // 公众号授予开放平台的权限不足
                    throw new WechatException(SalWechatErrorCode.WECHAT_AUTHORIZATION_DENIED);
                } else if (errorCode == 45015) {
                    // 48小时内无交互 无法主动给用户推送消息
                    throw new WechatException(SalWechatErrorCode.WECHAT_CANNOT_SENDMSG_TO_FANS);
                } else if (errorCode == 40003) {
                    // openId不合法
                    throw new WechatException(SalWechatErrorCode.WECHAT_ILLEGAL_OPENID);
                } else if (errorCode == 61003) {
                    //
                    throw new WechatException(SalWechatErrorCode.WECHAT_WRONG_COMPONET);
                } else if (errorCode == 61004) {
                    //服务器IP 不在微信开放平台配置的白名单内
                    log.error("ERROR - WechatJsonConverter convertJson json:{}", jsonData);
                    throw new WechatException(SalWechatErrorCode.WECHAT_API_RETURN_ERROR, "微信提示: " + errorMsg);
                } else if (errorCode == WechatErrorCode.ec40037.getCode()) {
                    throw new WechatException(SalWechatErrorCode.WECHAT_WRONG_TEMPLATE_ID);
                } else if (errorCode == WechatErrorCode.ec45047.getCode()) {
                    throw new WechatException(SalWechatErrorCode.MESSAGE_PUSH_TO_WECHAT_FAIL_45047);
                } else if (errorCode == WechatErrorCode.ec61023.getCode()) {
                    throw new WechatException(SalWechatErrorCode.WECHAT_REFRESH_TOKEN_IS_INVALID);
                } else {
                    log.warn("ERROR - WechatJsonConverter convertJson json:{}", jsonData);
                }

                WechatErrorCode wechatEC = WechatErrorCode.getByCode(errorCode);
                if (wechatEC != null) {
                    errorMsg = wechatEC.getMsg();
                }

                throw new WechatException(SalWechatErrorCode.WECHAT_API_RETURN_ERROR, "微信提示: " + errorMsg);
            }

        } else {
            log.warn("ERROR - WechatJsonConverter convertJson json:{}", jsonData);
            throw new WebServiceException("微信接口调用返回空值");
        }
        return new WechatApiResponse(errorCode, errorMsg, jsonData);
    }

    private static String doPostJSON(String url, String jsonStr) {
        if (StringUtils.isBlank(url)) {
            return null;
        }
        try {
            StringEntity postEntity = new StringEntity(jsonStr, "utf-8");
            postEntity.setContentEncoding("utf-8");
            postEntity.setContentType("application/json");
            HttpPost httpPost = new HttpPost(url);
            httpPost.setEntity(postEntity);

            RequestConfig config = RequestConfig.custom().setConnectTimeout(60000).setSocketTimeout(15000).build();
            CloseableHttpClient httpClient = HttpClientBuilder.create().setDefaultRequestConfig(config).build();

            CloseableHttpResponse response = httpClient.execute(httpPost);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                httpPost.abort();
                throw new RuntimeException("HttpClient,error status code :" + statusCode);
            }
            HttpEntity entity = response.getEntity();
            String result = null;
            if (entity != null) {
                result = EntityUtils.toString(entity, "utf-8");
            }
            EntityUtils.consume(entity);
            response.close();
            return result;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

}
