package com.baijia.tianxiao.sal.wechat.impl;

import java.util.Date;

import com.baijia.tianxiao.enums.RedisKeyEnums;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import com.baijia.tianxiao.dal.commons.configs.GenericsConfiguration;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.api.JsAPISDKService;
import com.baijia.tianxiao.sal.wechat.dto.JSSDK.JsApiTicket;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.util.JSSDKUtils;
import com.baijia.tianxiao.util.GenericsUtils;
import com.baijia.tianxiao.util.TwoTuple;

import lombok.extern.slf4j.Slf4j;

/**
 * @author Rezar
 * @createDate :Jun 30, 2016 3:37:20 PM
 * @desc :
 */
@Service
@Slf4j
public class JsAPISDKServiceImpl implements JsAPISDKService {

    @Autowired
    private AuthorizationInfoService authorizationInfoService;
    @Autowired(required = false)
    private RedisTemplate<String, String> redisTemplate;

    @Override
    public JsApiTicket retrievalJsApiTicket(Integer orgId) {
        RedisConnection connection = null;
        AuthorizationInfo authorizationInfo = this.authorizationInfoService.refreshAccessToken(orgId);
        if (authorizationInfo == null) {
            log.info("current org can not retrieval the jsApiTicket cause by no bind");
            return null;
        }
        try {
            if (this.redisTemplate != null) {
                log.info("获取到redis连接");
                // 获取到redis的连接
                connection = this.redisTemplate.getConnectionFactory().getConnection();
                connection.select(WechatProperties.getRedisDB());
                String redisKey = generateKey(authorizationInfo);
                byte[] bs = connection.get(redisKey.getBytes());
                RedisSerializer<?> redisSerializer = redisTemplate.getDefaultSerializer();
                Object deserialize = redisSerializer.deserialize(bs);

                if (deserialize != null && !GenericsConfiguration.needForceReGetJsApiTicket()) {
                    JsApiTicket ticket = (JsApiTicket) deserialize;
                    log.info("JsApiTicket 过期时间 param:{}", ticket.getExpiresIn());

                    // 判断是否过期
                    boolean isexpire = isAccessTokenExpired(ticket.getCreateTime());
                    log.info("[JsApiTicket] isexpire param:{}", isexpire);
                    if (isexpire) {
                        return refreshTicket(connection, authorizationInfo);
                    }
                    log.info("JsApiTicket ticket param:{}, appId param:{}", ticket,
                        authorizationInfo.getAuthorizerAppId());
                    return ticket;
                } else {
                    log.info(
                        "can not find any jsApiTicket from redis or it is expired , will lock and refresh api_ticket");
                    return refreshTicket(connection, authorizationInfo);
                }
            } else {
                log.info("find a null instance for redisTemplate and can not retrieval the jsApiTicket");
            }
        } catch (Exception e) {
            log.error(
                "can not retrieval jsapitTicket cause by : {} and will return it without reset it to redis server ", e);
        } finally {
            if (connection != null) {
                try {
                    RedisConnectionUtils.releaseConnection(connection, redisTemplate.getConnectionFactory());
                } catch (Exception e) {
                    log.info("can not releaseConnection cause by : {} ", e.getCause());
                }
            }
        }
        return reGetTicket(authorizationInfo);
    }

    private JsApiTicket refreshTicket(RedisConnection connection, AuthorizationInfo authorizationInfo) {
        String lockKey = getLockKey(authorizationInfo);
        JsApiTicket jat = null;
        if (connection.setNX(lockKey.getBytes(), "".getBytes())) {
            try {
                connection.expire(lockKey.getBytes(), 20);
                jat = this.reGetTicket(authorizationInfo);
                if (GenericsUtils.isNullOrEmpty(jat.getJsApiTicket())) {
                    return null;
                }
                @SuppressWarnings("unchecked")
                RedisSerializer<JsApiTicket> redisSerializer =
                    (RedisSerializer<JsApiTicket>) redisTemplate.getDefaultSerializer();
                log.info("JsApiTicket jat param:{}", jat);
                connection.set(generateKey(authorizationInfo).getBytes(), redisSerializer.serialize(jat));
                connection.expire(generateKey(authorizationInfo).getBytes(), jat.getExpiresIn());
            } finally {
                // 解锁
                try {
                    connection.del(lockKey.getBytes());
                } catch (Exception e) {
                    log.info("can not del key cause by : {} , and need reset ", e.getCause());
                }
            }
        }
        return jat;
    }

    public JsApiTicket reGetTicket(AuthorizationInfo authorizationInfo) {
        JsApiTicket jat = null;
        String accessToken = authorizationInfo.getAuthorizerAccessToken();
        log.info("accessToken param:{}", accessToken);
        TwoTuple<String, Integer> apiTicket = JSSDKUtils.trievalJsApiTicket(accessToken);
        String apiTicketStr = apiTicket.first;
        Integer expiredTime = apiTicket.second;
        expiredTime = expiredTime == null ? JsAPISDKService.JS_API_EXPIRED : expiredTime;
        jat = new JsApiTicket();
        jat.setExpiresIn(expiredTime);
        jat.setJsApiTicket(apiTicketStr);
        jat.setOrgId(authorizationInfo.getOrgId());
        jat.setCreateTime(new Date());
        jat.setAppId(authorizationInfo.getAuthorizerAppId());
        return jat;
    }

    private String getLockKey(AuthorizationInfo authorizationInfo) {
        return RedisKeyEnums.CRM.LOCK_JSAPITICKET_1.getRedisKey() + authorizationInfo.getOrgId();
    }

    private String generateKey(AuthorizationInfo authorizationInfo) {
        return RedisKeyEnums.CRM.KEY_JSAPITICKET.getRedisKey() + authorizationInfo.getOrgId();
    }

    public boolean isAccessTokenExpired(Date createTime) {
        Date now = new Date();
        if ((now.getTime() / 1000 - createTime.getTime() / 1000) >= JsAPISDKService.JS_API_EXPIRED) {
            return true;
        }
        return false;
    }

}
