/**
 * Baijiahulian.com Inc. Copyright (c) 2014-2015 All Rights Reserved.
 */
package com.baijia.tianxiao.sal.wechat.impl;

import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.baijia.tianxiao.dal.wechat.dao.AuthorizationInfoDao;
import com.baijia.tianxiao.dal.wechat.dao.ComponentAccessTokenDao;
import com.baijia.tianxiao.dal.wechat.po.AuthorizationInfo;
import com.baijia.tianxiao.exception.BussinessException;
import com.baijia.tianxiao.exception.WechatException;
import com.baijia.tianxiao.sal.wechat.api.AuthorizationInfoService;
import com.baijia.tianxiao.sal.wechat.constant.SalWechatErrorCode;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.OpenPlatformInfoDto;
import com.baijia.tianxiao.sal.wechat.dto.wechatapi.WechatApiResponse;
import com.baijia.tianxiao.sal.wechat.helper.WechatProperties;
import com.baijia.tianxiao.sal.wechat.helper.openplat.OpenPlatApiHelper;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;

/**
 * @Title : AuthorizationInfoServiceImpl
 * @Description :
 * @Author : zhenyujian
 * @Date : 2015年12月3日 下午2:28:47
 */
@Slf4j
@Service
public class AuthorizationInfoServiceImpl implements AuthorizationInfoService {

    @Autowired
    private AuthorizationInfoDao authorizationInfoDao;
    @Autowired
    private ComponentAccessTokenDao componentAccessTokenDao;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    @Transactional(readOnly = true)
    @Override
    public AuthorizationInfo getByOrgId(Integer orgId) {
        return authorizationInfoDao.getByOrgId(orgId);
    }

    @Transactional(readOnly = true)
    @Override
    public AuthorizationInfo getByAuthorizerAppId(String appId) {
        return authorizationInfoDao.getByAuthorizerAppId(appId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public AuthorizationInfo addAuthorizationInfo(Integer orgId, String authCode) {
        // 参数
        OpenPlatformInfoDto openPlatformInfo = WechatProperties.getOpenPlatformInfo();
        String componentAccessToken =
            componentAccessTokenDao.getByAppId(openPlatformInfo.getAppId()).getComponentAccessToken();

        // 发送请求 获取授权信息
        AuthorizationInfo newInfo =
            OpenPlatApiHelper.requestAuthorizationInfo(openPlatformInfo.getAppId(), componentAccessToken, authCode);
        AuthorizationInfo info = authorizationInfoDao.getByAuthorizerAppId(newInfo.getAuthorizerAppId());

        if (info != null) {
            // 机构账号已绑定了其它公众号
            if (!info.getOrgId().equals(orgId)) {
                throw new WechatException(SalWechatErrorCode.ORG_ALREADY_BIND_WECHATAPP);
            }
        } else {
            // 授权公众号已绑定了其它机构账号
            info = authorizationInfoDao.getByOrgId(orgId);
            if (info != null && !info.getAuthorizerAppId().equals(newInfo.getAuthorizerAppId())) {
                throw new WechatException(SalWechatErrorCode.WECHATAPP_ALREADY_BIND_ORG);
            }
        }

        Date now = new Date();
        if (info == null) {
            info = newInfo;
            info.setOrgId(orgId);
            info.setCreateTime(now);
            info.setUpdateTime(now);
            authorizationInfoDao.save(info, true);
            log.info("wechat - AuthorizationInfoServiceImpl - addAuthorizationInfo - add - info:{}", info);
            return info;
        } else {
            info.setOrgId(orgId);
            info.setAuthorizerAppId(newInfo.getAuthorizerAppId());
            info.setAuthorizerAccessToken(newInfo.getAuthorizerAccessToken());
            info.setAuthorizerRefershToken(newInfo.getAuthorizerRefershToken());
            info.setFuncs(newInfo.getFuncs());
            info.setExpiresIn(newInfo.getExpiresIn());
            info.setUpdateTime(now);
            authorizationInfoDao.update(info);
            log.info("wechat - AuthorizationInfoServiceImpl - addAuthorizationInfo - update - info:{}", info);
            return info;
        }
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    @Override
    public AuthorizationInfo refreshAccessToken(Integer orgId) {
        return refreshAccessToken(orgId, false);
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    @Override
    public AuthorizationInfo forceRefreshAccessToken(Integer orgId) throws BussinessException {
        return refreshAccessToken(orgId, true);
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = Exception.class)
    @Override
    public AuthorizationInfo refreshAccessToken(String appId) throws BussinessException {
        AuthorizationInfo authorizationInfo = authorizationInfoDao.getByAuthorizerAppId(appId);
        if (authorizationInfo != null) {
            return refreshAccessToken(authorizationInfo.getOrgId(), false);
        }
        return null;
    }

    private AuthorizationInfo refreshAccessToken(Integer orgId, boolean force) {
        AuthorizationInfo info = authorizationInfoDao.getByOrgId(orgId);
        log.info("refreshToken for OrgId:{} and ret:{} ", orgId, info);

        if (info == null) {
            throw new BussinessException(SalWechatErrorCode.ORG_NUBIND_WECHATAPP);
        }

        String lock = "wechat.refresh.accesstoken.lock." + info.getAuthorizerAppId();
        if (force || info.isAccessTokenExpired()) {
            RedisConnection connection = redisTemplate.getConnectionFactory().getConnection();
            try {
                connection.select(WechatProperties.getRedisDB());

                if (connection.setNX(lock.getBytes(), "".getBytes())) {
                    // 加锁 需要保证只有一个线程在刷新token
                    connection.expire(lock.getBytes(), 20);

                    // 参数
                    OpenPlatformInfoDto openPlatformInfo = WechatProperties.getOpenPlatformInfo();
                    String componentAccessToken =
                        componentAccessTokenDao.getByAppId(openPlatformInfo.getAppId()).getComponentAccessToken();
                    String authorizerAppId = info.getAuthorizerAppId();
                    String authorizerRefreshToken = info.getAuthorizerRefershToken();

                    // 请求
                    WechatApiResponse response = OpenPlatApiHelper.requestAuthorizerAccessToken(
                        openPlatformInfo.getAppId(), componentAccessToken, authorizerAppId, authorizerRefreshToken);

                    // 操作结果
                    JSONObject rootNode = response.getRootJSONObj();

                    info.setAuthorizerAccessToken(rootNode.getString("authorizer_access_token"));
                    info.setAuthorizerRefershToken(rootNode.getString("authorizer_refresh_token"));
                    info.setExpiresIn(rootNode.getInt("expires_in"));
                    info.setUpdateTime(new Date());
                    info.setOrgId(info.findReallyOrgId());
                    log.info("wechat - AuthorizationInfoServiceImpl - refreshAccessToken - orgId:{}, result:{}", orgId,
                        info);
                    authorizationInfoDao.update(info, true);

                }
            } catch (WechatException e) {
                log.warn("wechat - refreshAccessToken Exception - orgId:{},AuthorizationInfo:{}", orgId, info);
                log.warn("wechat - refreshAccessToken Exception - e", e);
                throw e;
            } catch (Exception e) {
                log.warn("wechat - refreshAccessToken Exception - orgId:{},AuthorizationInfo:{}", orgId, info);
                log.error("wechat - refreshAccessToken Exception - e", e);
                throw e;
            } finally {
                // 解锁
                connection.del(lock.getBytes());
                RedisConnectionUtils.releaseConnection(connection, redisTemplate.getConnectionFactory());
            }
        }

            log.info("wechat - AuthorizationInfoServiceImpl - refreshAccessToken - orgId:{}, authorizationInfo:{}", orgId,
            info);
        return info;
    }

}
